/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.Iterator;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.TrueFileFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.exception.RuntimeIOException;

import static java.lang.String.format;

public class FileComparisonUtils {

    private static final IOFileFilter ALL_FILES_FILTER = TrueFileFilter.INSTANCE;

    public static boolean contentEquals(File fileOrDir1, File fileOrDir2) throws IOException {
        if (!fileOrDir1.exists())
            throw new FileNotFoundException(fileOrDir1.getPath());
        if (!fileOrDir2.exists())
            throw new FileNotFoundException(fileOrDir2.getPath());

        if (fileOrDir1.isFile()) {
            if (fileOrDir2.isFile()) {
                return FileUtils.contentEquals(fileOrDir1, fileOrDir2);
            } else {
                return false;
            }
        } else {
            if (!fileOrDir2.isFile()) {
                return directoryContentEquals(fileOrDir1, fileOrDir2);
            } else {
                return false;
            }
        }
    }

    private static boolean directoryContentEquals(File dir1, File dir2) {
        return Arrays.equals(getDirectoryChecksum(dir1), getDirectoryChecksum(dir2));
    }

    @SuppressWarnings("unchecked")
    private static byte[] getDirectoryChecksum(File dir) {
        // used only if debug logging is on
        long startTime = 0;
        if (LOG.isDebugEnabled()) {
            startTime = System.currentTimeMillis();
            LOG.debug(format("Starting to calculate directory checksum for '%s'", dir, startTime));
        }
        MessageDigest digest = getMd5Digest();
        for (Iterator<File> files = FileUtils.iterateFiles(dir, ALL_FILES_FILTER, ALL_FILES_FILTER); files.hasNext();) {
            File file = files.next();

            // add both the content *and* the name because we still want to trigger a change if only the name changes
            try {
                updateChecksumFromContent(digest, file);
            } catch (IOException e) {
                throw new RuntimeIOException(format("Unable to update message digest for file '%s'", file), e);
            }
            digest.update(file.getName().getBytes());
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug(format("Calculated directory checksum for '%s' in %dms", dir, System.currentTimeMillis() - startTime));
        }
        return digest.digest();
    }

    private static MessageDigest getMd5Digest() {
        try {
            return MessageDigest.getInstance("MD5");
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("Unable to create MD5 digest", e);
        }
    }

    private static void updateChecksumFromContent(MessageDigest digest, File file) throws IOException {
        InputStream fis = new FileInputStream(file);
        byte[] buffer = new byte[1024];
        int numRead;
        do {
            numRead = fis.read(buffer);
            if (numRead > 0) {
                digest.update(buffer, 0, numRead);
            }
        } while (numRead != -1);
        fis.close();
    }

    private static final Logger LOG = LoggerFactory.getLogger(FileComparisonUtils.class);

}
