package com.xebialabs.deployit.util;

import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.atomic.AtomicReference;

import javax.crypto.Cipher;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;

import com.xebialabs.deployit.conversion.Base64Coder;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;

import static com.google.common.base.Preconditions.checkState;
import static com.xebialabs.deployit.checks.Checks.checkArgument;

public class PasswordEncrypter {

    private static final Base64Coder BASE_64_CODER = new Base64Coder();

    private static final AtomicReference<PasswordEncrypter> instance = new AtomicReference<PasswordEncrypter>();

    private final SecretKey passwordEncryptionKey;

    private PasswordEncrypter(SecretKey passwordEncryptionKey) {
        this.passwordEncryptionKey = passwordEncryptionKey;
    }

    public static void init(final SecretKey secretKey) {
        PasswordEncrypter passwordEncrypter = new PasswordEncrypter(secretKey);
        instance.set(passwordEncrypter);
    }

    public static PasswordEncrypter getInstance() {
        checkState(instance.get() != null, "PasswordEncrypter not yet set, please construct one.");
        return instance.get();
    }

    public boolean isEncoded(String text) {
        try {
            BASE_64_CODER.decode(text.getBytes());
            return true;
        } catch (RuntimeException e) {
            return false;
        }
    }

    public boolean isEncrypted(String text) {
        if (isEncoded(text)) {
            byte[] decode = BASE_64_CODER.decode(text.getBytes());
            try {
                Cipher cipher = getCipher(Cipher.DECRYPT_MODE);
                cipher.doFinal(decode);

                return true;
            } catch (GeneralSecurityException e) {
                throw new IllegalStateException("Could not decrypt Base64 encoded password");
            }
        }
        return false;
    }

    public String encrypt(String plaintext) {
        checkArgument(plaintext != null, "The password is null");
        checkArgument(!plaintext.startsWith(BASE_64_CODER.identifier()), "The password contains the illegal sequence " + BASE_64_CODER.identifier());

        try {
            Cipher aes = getCipher(Cipher.ENCRYPT_MODE);
            return new String(BASE_64_CODER.encode(aes.doFinal(plaintext.getBytes())));
        } catch (GeneralSecurityException e) {
            throw new DeployitException("Could not encrypt the password.");
        }
    }

    public String ensureEncrypted(String text) {
        if (isEncrypted(text)) {
            return text;
        } else {
            return encrypt(text);
        }
    }

    public String decrypt(String encryptedText) {
        checkArgument(encryptedText != null, "The password is null");
        checkArgument(encryptedText.startsWith(BASE_64_CODER.identifier()), "The password should be Base64 encoded");

        try {
            Cipher aes = getCipher(Cipher.DECRYPT_MODE);
            return new String(aes.doFinal(BASE_64_CODER.decode(encryptedText.getBytes())));
        } catch (GeneralSecurityException e) {
            throw new DeployitException("Could not decrypt the password.");
        }
    }

    public String ensureDecrypted(String text) {
        if (isEncrypted(text)) {
            return decrypt(text);
        } else {
            return text;
        }
    }

    private Cipher getCipher(int mode) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException {
        Cipher aes = Cipher.getInstance(passwordEncryptionKey.getAlgorithm());
        aes.init(mode, passwordEncryptionKey);
        return aes;
    }

}
