package com.xebialabs.deployit.core.upgrade;

import java.security.Principal;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import javax.jcr.*;
import javax.jcr.nodetype.NodeType;
import javax.jcr.security.AccessControlEntry;
import javax.jcr.security.AccessControlManager;
import javax.jcr.security.AccessControlPolicy;
import javax.jcr.security.Privilege;
import org.apache.jackrabbit.api.security.JackrabbitAccessControlEntry;
import org.apache.jackrabbit.api.security.JackrabbitAccessControlList;
import org.apache.jackrabbit.core.security.principal.EveryonePrincipal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Predicate;
import com.google.common.base.Splitter;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.*;

import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.jcr.JcrUtils;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot;
import com.xebialabs.deployit.repository.core.Directory;
import com.xebialabs.deployit.repository.core.Securable;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.server.api.repository.RawRepository;
import com.xebialabs.deployit.server.api.upgrade.Upgrade;
import com.xebialabs.deployit.server.api.upgrade.UpgradeException;
import com.xebialabs.deployit.server.api.upgrade.Version;
import com.xebialabs.deployit.upgrade.RawRepositoryImpl;
import com.xebialabs.deployit.util.Tuple;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.transform;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Multimaps.index;
import static com.google.common.collect.Sets.filter;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.*;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;
import static com.xebialabs.deployit.repository.JcrPathHelper.getIdFromAbsolutePath;
import static com.xebialabs.deployit.security.permission.DeployitPermissions.*;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_REPO;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.READ;
import static java.lang.String.format;

public class Deployit37Security extends Upgrade {

    public static final Joiner PRINCIPAL_JOINER = Joiner.on(",").skipNulls();
    private static final Set<Permission> RELATED_ACL_PERMISSIONS = newHashSet(DEPLOY_INITIAL, DEPLOY_UPGRADE, UNDEPLOY);
    private static final Set<Permission> EX_GLOBAL_PERMISSIONS = newHashSet(READ, IMPORT_INITIAL, IMPORT_UPGRADE, DEPLOY_INITIAL, DEPLOY_UPGRADE, UNDEPLOY, EDIT_REPO);
    private static final Function<Value, String> VALUE_TO_STRING = new Function<Value, String>() {
        @Override
        public String apply(Value input) {
            try {
                return input.getString();
            } catch (RepositoryException e) {
                logger.error("Couldn't convert value to string", e);
                throw new UpgradeException("Couldn't convert value to string");
            }
        }
    };

    private static final String GLOBAL = "<global>";
    private static final Splitter PATH_SPLITTER = Splitter.on("/").omitEmptyStrings();
    private static final Joiner PATH_JOINER = Joiner.on("/");
    private static final String DIRECTORY_PREFIX = "migration-dir-";

    @Override
    public boolean doUpgrade(RawRepository repository) throws UpgradeException {
        logger.info("*** Running Deployit 3.7.0 Security Upgrade ***");
        Node securityNode = repository.read(JcrConstants.SECURITY_NODE_ID);
        Map<String, HashMultimap<Permission, String>> newPermissionStructure;
        try {
            Map<String, ListMultimap<Permission, String>> analyzedPermissionStructure = analyzePermissionStructure(securityNode);

            logger.info("This is the analyzed permission structure:");
            logPermissionStructure(analyzedPermissionStructure);

            removeNonExistingNodesFromAnalyzedStructure(analyzedPermissionStructure, repository);
            removeDeniesFromAnalyzedStructure(analyzedPermissionStructure, repository);

            newPermissionStructure = filterDuplicateGrants(analyzedPermissionStructure);
            removeNonTopLevelPermissions(newPermissionStructure);
            fixinNonInheritanceInNewStructure(newPermissionStructure);


            Map<String, String> nodeToDirmapping = mapToDirectories(newPermissionStructure);
            remapPermissionStructure(newPermissionStructure, nodeToDirmapping);

            mapGlobalPermissionsToConfigurationItems(newPermissionStructure);
            addImportRemoveToApplications(newPermissionStructure);
            logger.info("This will be the new Permission Structure:");
            logPermissionStructure(newPermissionStructure);

            clearAllAcls(analyzedPermissionStructure, repository);
            createNewRepositoryStructure(nodeToDirmapping, repository);
            makeRootNodesVersionable(repository);

            Set<String> principals = gatherAllKnownPrincipals(newPermissionStructure);
            logger.info("Found the following principals known to Deployit: [{}]", principals);

            List<Role> roles = convertToRoles(principals);
            writeRoles(roles, repository);
            writeRoleAssignments(roles, repository);

            writeNewPermissions(newPermissionStructure, roles, repository, securityNode);

        } catch (RepositoryException e) {
            logger.error("Exception while upgrading security", e);
            throw new UpgradeException("Caught an exception while upgrading to 3.7 Security, please send the server log.");
        }

        logger.info("*** Finished Upgrade ***");
        return true;
    }

    private void makeRootNodesVersionable(RawRepository repository) throws RepositoryException {
        for (ConfigurationItemRoot root : EnumSet.complementOf(EnumSet.of(NESTED))) {
            String absolutePathFromId = getAbsolutePathFromId(root.getRootNodeName());
            if (exists(repository, absolutePathFromId)) {
                Node node = repository.read(absolutePathFromId);
                node.addMixin(NodeType.MIX_VERSIONABLE);
            }
        }
    }

    private void logPermissionStructure(Map<String, ? extends Multimap<Permission, String>> permissionStructure) {
        ArrayList<String> nodes = newArrayList(permissionStructure.keySet());
        Collections.sort(nodes);

        StringBuilder b = new StringBuilder();
        for (String node : nodes) {
            b.append(format("On node [%s]:\n", node));
            for (Map.Entry<Permission, Collection<String>> pe : permissionStructure.get(node).asMap().entrySet()) {
                b.append(format("\t%s=%s\n", pe.getKey().getPermissionName(), pe.getValue()));
            }
            b.append("\n");
        }

        logger.info("\n{}", b);
    }

    private Map<String, ListMultimap<Permission, String>> analyzePermissionStructure(Node securityNode) throws RepositoryException {
        logger.info("Analyzing the existing permission structure");
        final LoadingCache<String, ListMultimap<Permission, String>> analysis = CacheBuilder.newBuilder().build(CacheLoader.from(new Function<String, ListMultimap<Permission, String>>() {
            public ListMultimap<Permission, String> apply(String input) {
                return ArrayListMultimap.create();
            }
        }));
        JcrUtils.forEachNonJcrProperty(securityNode, new JcrUtils.Callback<Property>() {
            public void apply(Property input) throws RepositoryException {
                String user = input.getName();
                List<Tuple<Permission, String>> permissionsGranted = transformValuesToPermissions(input);
                logger.debug("Found permissions [{}] granted to [{}]", permissionsGranted, user);
                for (Tuple<Permission, String> permissionNodeTuple : permissionsGranted) {
                    String nodePath = permissionNodeTuple.getB();
                    Permission permission = permissionNodeTuple.getA();
                    String absolutePathFromId = nodePath.equals(GLOBAL) ? nodePath : getAbsolutePathFromId(nodePath);
                    logger.debug("Adding to migration: permission [{}] on [{}] granted to [{}]", new Object[]{permission, absolutePathFromId, user});
                    analysis.getUnchecked(absolutePathFromId).put(permission, user);
                }
                input.remove();
            }
        });
        return analysis.asMap();
    }

    private void removeNonExistingNodesFromAnalyzedStructure(Map<String, ListMultimap<Permission, String>> analyzedPermissionStructure, RawRepository repository) {
        logger.info("Removing non-existing nodes from the analyzed permission structure");
        List<String> toBeRemoved = newArrayList();
        for (String nodePath : analyzedPermissionStructure.keySet()) {
            if (nodePath.equals(GLOBAL)) continue;

            boolean exists;
            try {
                exists = (repository.read(nodePath) != null);
            } catch (RuntimeException re) {
                exists = false;
            }

            if (!exists) {
                toBeRemoved.add(nodePath);
            }
        }

        for (String nodePath : toBeRemoved) {
            logger.info("Removing node [{}] with permissions [{}] from migration as it no longer exists.", nodePath, analyzedPermissionStructure.get(nodePath));
            analyzedPermissionStructure.remove(nodePath);
        }
    }

    private void removeDeniesFromAnalyzedStructure(Map<String, ListMultimap<Permission, String>> analyzedPermissionStructure, RawRepository repository) throws RepositoryException {
        logger.info("Cleaning denies from the analyzed permission structure");
        for (String node : analyzedPermissionStructure.keySet()) {
            ListMultimap<Permission, String> permissionToUserMap = analyzedPermissionStructure.get(node);

            if (!permissionToUserMap.containsKey(READ)) {
                logger.debug("Node [{}] does not have any read permissions", node);
                continue;
            }

            AccessControlManager accessControlManager = getAccessControlManager(repository);
            String path = node;
            if (node.equals(GLOBAL)) {
                path = "/";
            }
            AccessControlPolicy[] policies = accessControlManager.getPolicies(path);
            for (AccessControlPolicy policy : policies) {
                if (!(policy instanceof JackrabbitAccessControlList)) {
                    continue;
                }

                JackrabbitAccessControlList acl = (JackrabbitAccessControlList) policy;
                for (AccessControlEntry ace : acl.getAccessControlEntries()) {
                    if (!(ace instanceof JackrabbitAccessControlEntry)) {
                        continue;
                    }

                    boolean allowed = ((JackrabbitAccessControlEntry) ace).isAllow();
                    Privilege[] privileges = ace.getPrivileges();
                    if (!allowed && privileges.length == 1 && privileges[0].getName().equals("jcr:read")) {
                        Principal principal = ace.getPrincipal();
                        permissionToUserMap.get(READ).remove(principal.getName());
                        logger.warn("Not migrating denied READ permission granted to [{}] on [{}]", principal.getName(), node);
                    }
                }
            }
        }
    }

    private Map<String, HashMultimap<Permission, String>> filterDuplicateGrants(Map<String, ListMultimap<Permission, String>> analyzedPermissionStructure) {
        logger.info("Filtering duplicate principals from grants");
        Map<String, HashMultimap<Permission, String>> filtered = newHashMap();

        for (Map.Entry<String, ListMultimap<Permission, String>> e : analyzedPermissionStructure.entrySet()) {
            filtered.put(e.getKey(), HashMultimap.create(e.getValue()));
        }

        return filtered;
    }

    private void removeNonTopLevelPermissions(Map<String, HashMultimap<Permission, String>> newPermissionStructure) {
        logger.info("Removing all permissions not granted to the first 2 levels of the repository.");

        Set<String> deepNodes = newHashSet(Sets.filter(newPermissionStructure.keySet(), new Predicate<String>() {
            @Override
            public boolean apply(String input) {
                return Iterables.size(PATH_SPLITTER.split(input)) > 2;
            }
        }));

        for (String deepNode : deepNodes) {
            HashMultimap<Permission, String> permissions = newPermissionStructure.get(deepNode);
            logger.warn("Not migrating permissions on node [{}] ({})", deepNode, permissions);
            newPermissionStructure.remove(deepNode);
        }
    }

    private void fixinNonInheritanceInNewStructure(Map<String, HashMultimap<Permission, String>> newPermissionStructure) {
        logger.info("Consolidating the new permission structure");

        ArrayList<String> nodes = newArrayList(newPermissionStructure.keySet());
        Collections.sort(nodes);
        logger.debug("Sorted the nodes in parent->child order: {}", nodes);

        Stack<String> previousNodes = new Stack<String>();
        for (String node : nodes) {
            while (!previousNodes.isEmpty()) {
                String previousNode = previousNodes.peek();
                if (node.startsWith(previousNode)) {
                    logger.debug("Copying read/repo#edit permissions from [{}] to [{}]", previousNode, node);
                    Multimap<Permission, String> parentPermissions = newPermissionStructure.get(previousNode);
                    Multimap<Permission, String> myPermissions = newPermissionStructure.get(node);
                    copyPermissions(parentPermissions, myPermissions, READ);
                    copyPermissions(parentPermissions, myPermissions, EDIT_REPO);
                    break;
                } else {
                    previousNodes.pop();
                }
            }
            previousNodes.push(node);
        }
    }

    private Map<String, String> mapToDirectories(Map<String, HashMultimap<Permission, String>> newPermissionStructure) {
        logger.info("Separating permission structure to groups");
        Map<String, HashMultimap<Permission, String>> onlySubNodePermissions = Maps.filterKeys(newPermissionStructure, new Predicate<String>() {
            public boolean apply(String input) {
                return input.indexOf("/", 1) > -1;
            }
        });

        ListMultimap<String, String> lookupRootToNode = index(onlySubNodePermissions.keySet(), new Function<String, String>() {
            public String apply(String input) {
                return PATH_SPLITTER.split(input).iterator().next();
            }
        });
        logger.trace("Indexed nodes from permission structure to roots: {}", lookupRootToNode);


        logger.debug("Building mapping from permissions -> directories");
        Map<HashMultimap<Permission, String>, String> lookupPermissionToDirectoryMap = newHashMap();
        int groupNr = 1;
        HashSet<HashMultimap<Permission, String>> uniqueSubNodeOnlyPermissions = newHashSet(onlySubNodePermissions.values());
        for (HashMultimap<Permission, String> permissions : uniqueSubNodeOnlyPermissions) {
            lookupPermissionToDirectoryMap.put(permissions, DIRECTORY_PREFIX + groupNr++);
        }
        logger.trace("Permission -> directory: {}", lookupPermissionToDirectoryMap);

        Map<String, String> nodeToGroup = newHashMap();
        for (Map.Entry<String, Collection<String>> e : lookupRootToNode.asMap().entrySet()) {
            String rootId = getAbsolutePathFromId(e.getKey());
            Collection<String> nodesInRoot = e.getValue();
            for (String node : nodesInRoot) {
                HashMultimap<Permission, String> permissions = onlySubNodePermissions.get(node);
                String dirName = lookupPermissionToDirectoryMap.get(permissions);
                if (dirName != null) {
                    nodeToGroup.put(node, PATH_JOINER.join(rootId, dirName));
                }
            }
        }

        logger.debug("Mapped nodes to directories: {}", nodeToGroup);

        return nodeToGroup;
    }

    private void remapPermissionStructure(Map<String, HashMultimap<Permission, String>> newPermissionStructure, Map<String, String> groupMapping) {
        logger.info("Remapping the permission structure to groups.");
        for (String node : newHashSet(newPermissionStructure.keySet())) {
            String group = groupMapping.get(node);
            if (group != null) {
                HashMultimap<Permission, String> permissions = newPermissionStructure.get(node);
                newPermissionStructure.remove(node);
                newPermissionStructure.put(group, permissions);
            }
        }
    }

    private void mapGlobalPermissionsToConfigurationItems(Map<String, HashMultimap<Permission, String>> newPermissionStructure) {
        logger.info("Mapping old global permissions to configuration items.");
        HashMultimap<Permission, String> permissionPrincipals = newPermissionStructure.get(GLOBAL);
        if(permissionPrincipals == null) {
            permissionPrincipals = HashMultimap.create();
        }
        Set<String> nodes = filter(newPermissionStructure.keySet(), new Predicate<String>() {
            @Override
            public boolean apply(String input) {
                return !input.equals(GLOBAL);
            }
        });
        for (Permission permission : EX_GLOBAL_PERMISSIONS) {
            if (permissionPrincipals.containsKey(permission)) {
                for (String node : nodes) {
                    if (permission.isApplicableTo(getIdFromAbsolutePath(node))) {
                        newPermissionStructure.get(node).putAll(permission, permissionPrincipals.get(permission));
                    }
                }
            }
            permissionPrincipals.removeAll(permission);
        }
    }

    private void addImportRemoveToApplications(Map<String, HashMultimap<Permission, String>> newPermissionStructure) {
        logger.info("Adding import#remove to Applications with repo#edit");
        Set<Map.Entry<String,HashMultimap<Permission,String>>> applications = filter(newPermissionStructure.entrySet(), new Predicate<Map.Entry<String, HashMultimap<Permission, String>>>() {
            @Override
            public boolean apply(Map.Entry<String, HashMultimap<Permission, String>> input) {
                boolean isApp = getIdFromAbsolutePath(input.getKey()).startsWith(APPLICATIONS.getRootNodeName());
                boolean hasEditRepo = input.getValue().containsKey(EDIT_REPO);
                return isApp && hasEditRepo;
            }
        });

        for (Map.Entry<String, HashMultimap<Permission, String>> app : applications) {
            Set<String> principals = newHashSet(app.getValue().get(EDIT_REPO));
            logger.debug("Add import#remove to [{}] for principals [{}]", app.getKey(), principals);
            app.getValue().putAll(IMPORT_REMOVE, principals);
        }
    }



    private void clearAllAcls(Map<String, ListMultimap<Permission, String>> analyzedPermissionStructure, RawRepository repository) throws RepositoryException {
        logger.info("Clearing out old permissions!");

        for (Map.Entry<String, ListMultimap<Permission, String>> e : analyzedPermissionStructure.entrySet()) {
            String absoluteNodePath = e.getKey();
            ListMultimap<Permission, String> permissionToUser = e.getValue();
            if (!absoluteNodePath.equals(GLOBAL)) {
                clearNodeAcl(repository, absoluteNodePath, permissionToUser);
            }
        }

        clearRoots(repository);
        clearAcl(JcrConstants.TASKS_NODE_ID, getAccessControlManager(repository));
        clearAcl(JcrConstants.CONFIGURATION_NODE_ID, getAccessControlManager(repository));
        clearRepositoryRoot(repository);

    }

    private void clearNodeAcl(RawRepository repository, String absoluteNodePath, ListMultimap<Permission, String> permissionToUser) throws RepositoryException {
        clearAcl(absoluteNodePath, getAccessControlManager(repository));
        for (Permission permission : permissionToUser.keySet()) {
            if (hasRelatedAcls(permission)) {
                clearRelatedAcls(absoluteNodePath, repository);
            }
        }
    }

    private void clearRelatedAcls(String absoluteAffectedNodePath, RawRepository repository) throws RepositoryException {
        logger.info("Clearing related ACL from [{}]", absoluteAffectedNodePath);
        checkArgument(absoluteAffectedNodePath.contains(ENVIRONMENTS.getRootNodeName()), "Only environments have a related ACL (%s)", absoluteAffectedNodePath);
        Node read = repository.read(absoluteAffectedNodePath);
        List<String> members = newArrayList();
        List<String> dictionaries = newArrayList();
        if (read.hasProperty("members")) {
            members = transform(newArrayList(read.getProperty("members").getValues()), VALUE_TO_STRING);
        }
        if (read.hasProperty("dictionaries")) {
            dictionaries = transform(newArrayList(read.getProperty("dictionaries").getValues()), VALUE_TO_STRING);
        }
        for (String memberUuid : members) {
            String member = ((RawRepositoryImpl) repository).getSession().getNodeByIdentifier(memberUuid).getPath();
            logger.debug("Clearing ACL from member [{}] of environment [{}]", member, absoluteAffectedNodePath);
            clearAcl(member, getAccessControlManager(repository));
        }

        for (String dictUuid : dictionaries) {
            String dict = ((RawRepositoryImpl) repository).getSession().getNodeByIdentifier(dictUuid).getPath();
            logger.debug("Clearing ACL from dictionary [{}] of environment [{}]", dict, absoluteAffectedNodePath);
            clearAcl(dict, getAccessControlManager(repository));
        }
    }

    private void createNewRepositoryStructure(Map<String, String> nodeToDirMapping, RawRepository repository) throws RepositoryException {
        logger.info("Creating the new Repository structure.");
        String directoryType = Type.valueOf(Directory.class).toString();
        for (String dir : newHashSet(nodeToDirMapping.values())) {
            Node node = repository.create(dir);
            if (node.getIndex() != 1) {
                node.remove();
                throw new IllegalStateException("The impossible has happened, you already have a node called [" + dir + "] in your repository. Please contact deployit-support@xebialabs.com");
            }
            node.addMixin(CONFIGURATION_ITEM_NODETYPE_NAME);
            node.addMixin(NodeType.MIX_REFERENCEABLE);
            node.addMixin(NodeType.MIX_VERSIONABLE);
            node.setProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, directoryType);
            logger.debug("Created a [{}] at [{}]", directoryType, dir);
        }

        for (Map.Entry<String, String> entry : nodeToDirMapping.entrySet()) {
            String node = entry.getKey();
            String dir = entry.getValue();
            String newLocation = PATH_JOINER.join(dir, Iterables.getLast(PATH_SPLITTER.split(node)));

            logger.debug("Moving [{}] to new location [{}]", node, newLocation);
            ((RawRepositoryImpl) repository).getSession().move(node, newLocation);
        }
    }

    private void writeNewPermissions(Map<String, HashMultimap<Permission, String>> newPermissionStructure, List<Role> roles, RawRepository repository, Node securityNode) throws RepositoryException {
        logger.info("Writing the new permissions!");
        for (Map.Entry<String, HashMultimap<Permission, String>> nodePermission : newPermissionStructure.entrySet()) {
            if (nodePermission.getKey().equals(GLOBAL)) {
                logger.info("Writing global permissions.");
                writeNewPermissions(nodePermission.getValue(), roles, securityNode);
            } else {
                writeNewNodePermissions(nodePermission.getKey(), nodePermission.getValue(), roles, repository);
            }
        }
    }

    private void writeNewNodePermissions(String absoluteNodePath, HashMultimap<Permission, String> permissionToUser, List<Role> roles, RawRepository repository) throws RepositoryException {
        logger.info("Writing new permissions for node [{}]", absoluteNodePath);
        Node read = repository.read(absoluteNodePath);
        writeNewPermissions(permissionToUser, roles, read);
    }

    private void writeNewPermissions(HashMultimap<Permission, String> permissionToUser, List<Role> roles, Node securityNode) throws RepositoryException {
        final ImmutableMap<String, Role> roleLookup = Maps.uniqueIndex(roles, new Function<Role, String>() {
            @Override
            public String apply(Role input) {
                return input.getName();
            }
        });
        Map<String, String> toWrite = newHashMap();
        for (Map.Entry<Permission, Collection<String>> entry : permissionToUser.asMap().entrySet()) {
            toWrite.put(entry.getKey().getPermissionName(), PRINCIPAL_JOINER.join(Iterables.transform(entry.getValue(), new Function<String, Integer>() {
                public Integer apply(String input) {
                    Role role = roleLookup.get(input);
                    return role.getId();
                }
            })));
        }

        JcrUtils.writeMap(securityNode, Securable.SECURITY_PERMISSIONS_PROPERTY, toWrite);
    }

    private void writeRoleAssignments(List<Role> roles, RawRepository repository) throws RepositoryException {
        logger.info("Writing initial role assignments");
        Node node = repository.create(JcrConstants.ROLE_ASSIGNMENTS_NODE_ID);
        for (Role role : roles) {
            node.setProperty(role.getName(), role.getId().toString());
        }
    }

    private List<Role> convertToRoles(Set<String> principals) {
        logger.info("Converting all principals to roles");
        final AtomicInteger counter = new AtomicInteger(1);
        return newArrayList(Iterables.transform(principals, new Function<String, Role>() {
            public Role apply(String input) {
                Role role = new Role(counter.getAndIncrement(), input);
                logger.debug("Giving principal [{}] role id [{}]", input, role.getId());
                return role;
            }
        }));
    }

    private void writeRoles(List<Role> roles, RawRepository repository) throws RepositoryException {
        logger.info("Writing new Roles");
        Node roleNode = repository.create(JcrConstants.ROLES_NODE_ID);

        for (Role role : roles) {
            roleNode.setProperty(role.getId().toString(), role.getName());
        }
    }

    private Set<String> gatherAllKnownPrincipals(Map<String, HashMultimap<Permission, String>> newPermissionStructure) throws RepositoryException {
        logger.info("Extracting all known principals");
        Set<String> principals = newHashSet();
        for (Map.Entry<String, HashMultimap<Permission, String>> entry : newPermissionStructure.entrySet()) {
            logger.info("Extracting known principals from [{}]", entry.getKey());
            HashMultimap<Permission, String> value = entry.getValue();
            Collection<Collection<String>> ccprincipals = value.asMap().values();
            for (Collection<String> cprincipals : ccprincipals) {
                principals.addAll(cprincipals);
            }
        }
        return principals;
    }

    @Override
    public Version upgradeVersion() {
        return Version.valueOf("deployit", "3.7.0");
    }

/*
     * Helper methods.
     */

    private static boolean hasRelatedAcls(Permission permission) {
        return RELATED_ACL_PERMISSIONS.contains(permission);
    }

    private static List<Tuple<Permission, String>> transformValuesToPermissions(Property property) throws RepositoryException {
        Value[] values = property.getValues();
        return transform(newArrayList(values), new Function<Value, Tuple<Permission, String>>() {
            public Tuple<Permission, String> apply(Value input) {
                try {
                    return decodePermissionString(input.getString());
                } catch (RepositoryException e) {
                    logger.error("Exception while migrating security: ", e);
                    throw new UpgradeException("Please read the server log to see what went wrong with migration of security.");
                }
            }
        });
    }

    private static Tuple<Permission, String> decodePermissionString(String permissionString) {
        for (Permission permission : Permission.getAll()) {
            if (permissionString.startsWith(permission.getPermissionName())) {
                return decodePermissionString(permissionString, permission);
            }
        }
        throw new RuntimeException("PermissionString " + permissionString + " could not be decoded");
    }

    private static Tuple<Permission, String> decodePermissionString(String permissionString, Permission permission) {
        int permissionNameCiSeparatorIndex = permission.getPermissionName().length() + 1;
        if (permissionNameCiSeparatorIndex >= permissionString.length()) {
            return new Tuple<Permission, String>(permission, GLOBAL);
        }
        String ciName = permissionString.substring(permissionNameCiSeparatorIndex).replace("$", "/");
        return new Tuple<Permission, String>(permission, ciName);
    }

    private static void copyPermissions(Multimap<Permission, String> parentPermissions, Multimap<Permission, String> myPermissions, Permission p) {
        if (parentPermissions.containsKey(p)) {
            myPermissions.putAll(p, parentPermissions.get(p));
        }
    }

    private static AccessControlManager getAccessControlManager(RawRepository repository) throws RepositoryException {
        Session session = ((RawRepositoryImpl) repository).getSession();
        return session.getAccessControlManager();
    }

    private static void clearAcl(String absoluteAffectedNodePath, AccessControlManager accessControlManager) throws RepositoryException {
        logger.info("Clearing ACL from [{}]", absoluteAffectedNodePath);
        AccessControlPolicy[] policies = accessControlManager.getPolicies(absoluteAffectedNodePath);
        for (AccessControlPolicy policy : policies) {
            if (!(policy instanceof JackrabbitAccessControlList))
                continue;

            JackrabbitAccessControlList acl = (JackrabbitAccessControlList) policy;
            for (AccessControlEntry ace : acl.getAccessControlEntries()) {
                acl.removeAccessControlEntry(ace);
            }

            accessControlManager.setPolicy(absoluteAffectedNodePath, policy);
        }
    }

    private static void clearRoots(RawRepository repository) throws RepositoryException {
        logger.info("Clearing ACL on Root nodes.");
        for (ConfigurationItemRoot root : EnumSet.complementOf(EnumSet.of(NESTED))) {
            String rootPath = getAbsolutePathFromId(root.getRootNodeName());
            if (exists(repository, rootPath)) {
                clearAcl(rootPath, getAccessControlManager(repository));
            }
        }
    }

    private static boolean exists(RawRepository repository, String rootPath) throws RepositoryException {
        return ((RawRepositoryImpl) repository).getSession().nodeExists(rootPath);
    }

    private static void clearRepositoryRoot(RawRepository repository) throws RepositoryException {
        logger.info("Clearing the repository root...");
        clearAcl("/", getAccessControlManager(repository));
        grantEveryoneReadAccess("/", getAccessControlManager(repository));
    }

    private static void grantEveryoneReadAccess(String absPath, AccessControlManager accessControlManager) throws RepositoryException {
        for (AccessControlPolicy policy : accessControlManager.getPolicies(absPath)) {
            if (!(policy instanceof JackrabbitAccessControlList)) {
                continue;
            }

            JackrabbitAccessControlList acl = (JackrabbitAccessControlList) policy;
            Privilege privilege = accessControlManager.privilegeFromName(Privilege.JCR_READ);
            acl.addAccessControlEntry(EveryonePrincipal.getInstance(), new Privilege[] {privilege});
            accessControlManager.setPolicy("/", policy);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(Deployit37Security.class);
}
