package com.xebialabs.deployit.core.upgrade;

import java.io.IOException;
import java.io.InputStream;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.util.Collection;
import java.util.List;
import javax.jcr.Node;
import javax.jcr.RepositoryException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.io.ByteSource;
import com.google.common.io.ByteStreams;

import com.xebialabs.deployit.io.Exploder;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.FolderArtifact;
import com.xebialabs.deployit.server.api.repository.RawRepository;
import com.xebialabs.deployit.server.api.upgrade.Upgrade;
import com.xebialabs.deployit.server.api.upgrade.UpgradeException;
import com.xebialabs.deployit.server.api.upgrade.Version;
import com.xebialabs.deployit.util.DevNull;
import com.xebialabs.deployit.util.JavaCryptoUtils;

public class Deployit394Checksums extends Upgrade {
    @Override
    public boolean doUpgrade(RawRepository repository) throws UpgradeException {
        Type type = Type.valueOf(DeployableArtifact.class);
        Collection<Type> subtypes = DescriptorRegistry.getSubtypes(type);
        for (Type subtype : subtypes) {
            try {
                upgradeNodesOfType(repository, subtype);
            } catch (RepositoryException | IOException e) {
                throw new UpgradeException("Upgrade failed due to", e);
            }
        }
        return true;
    }

    private void upgradeNodesOfType(RawRepository repository, Type type) throws RepositoryException, IOException {
        if (!type.getDescriptor().isVirtual()) {
            logger.info("Reading all nodes of type [{}]", type);
            List<Node> nodesByType = repository.findNodesByType(type);
            for (Node node : nodesByType) {
                upgradeNode(type, node);
            }
        }
    }

    private void upgradeNode(Type type, final Node node) throws RepositoryException, IOException {
        if (node.hasProperty("checksum")) {
            logger.debug("Checking whether checksum of [{}] is broken", node.getPath());
            String checksum = node.getProperty("checksum").getString();
            if (isPossiblyCorrupt(checksum)) {
                logger.debug("Checksum [{}] of node [{}] needs verification", checksum, node.getPath());
                String realChecksum = calculateRealChecksum(type, node);
                if (isReallyCorrupt(checksum, realChecksum)) {
                    logger.info("Fixing checksum [{}] of node [{}] to [{}]", checksum, node.getPath(), realChecksum);
                    node.setProperty("checksum", realChecksum);
                }

            }
        } else {
            logger.debug("Skipping node [{}] as it has no checksum", node.getPath());
        }
    }

    private String calculateRealChecksum(Type type, Node node) throws IOException {
        final MessageDigest sha1 = JavaCryptoUtils.getSha1();
        ByteSource inputSupplier = getByteSource(node);
        if (type.isSubTypeOf(Type.valueOf(FolderArtifact.class))) {
            Exploder.calculateCheckSum(inputSupplier, sha1);
        } else {
            try(InputStream input = inputSupplier.openStream()) {
                ByteStreams.copy(new DigestInputStream(input, sha1), new DevNull());
            }
        }
        return JavaCryptoUtils.digest(sha1);
    }

    private ByteSource getByteSource(final Node node) {
        return new ByteSource() {
            @Override
            public InputStream openStream() throws IOException {
                try {
                    return node.getProperty(JcrConstants.DATA_PROPERTY_NAME).getBinary().getStream();
                } catch (RepositoryException e) {
                    throw new IOException(e);
                }
            }
        };

    }

    @VisibleForTesting
    boolean isReallyCorrupt(String checksum, String realChecksum) {
        char[] checksumChars = checksum.toCharArray();
        char[] realChecksumChars = realChecksum.toCharArray();
        for (int i = 0; i < checksumChars.length; i++) {
            char checksumChar = checksumChars[i];
            char realChecksumChar = realChecksumChars[i];
            if (checksumChar != realChecksumChar) {
                if (realChecksumChar == '0') {
                    realChecksumChar = realChecksumChars[i + 1];
                    // Fast return, if we encounter a '0' in the new checksum and the next character matches the previous checksum, then mark as corrupt.
                    return checksumChar == realChecksumChar;
                } else {
                    return false;
                }
            }
        }
        return false;
    }

    @VisibleForTesting
    boolean isPossiblyCorrupt(String checksum) {
        return checksum.matches("^[0-9A-Fa-f]{20,39}$");
    }

    @Override
    public Version upgradeVersion() {
        return Version.valueOf("deployit", "3.9.4");
    }

    private static final Logger logger = LoggerFactory.getLogger(Deployit394Checksums.class);

}
