package com.xebialabs.deployit.service.controltask

import java.util

import com.xebialabs.deployit.checks.Checks._
import com.xebialabs.deployit.deployment.planner._
import com.xebialabs.deployit.engine.tasker._
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, MethodDescriptor, Type}
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, Parameters}
import com.xebialabs.deployit.repository.WorkDir
import com.xebialabs.deployit.security.Permissions
import com.xebialabs.deployit.task.TaskMetadata.{CLOUD_ENVIRONMENT_ID, CLOUD_ENVIRONMENT_TEMPLATE_ID, CLOUD_OPERATION, CONTROL_TASK_TARGET_CI, TASK_NAME, _}
import com.xebialabs.deployit.task.archive.NodeNames
import com.xebialabs.deployit.task.{TaskType, WorkdirCleanerTrigger}
import com.xebialabs.xlplatform.satellite.{Satellite, SatelliteAware}
import org.springframework.stereotype.Component

import scala.collection.convert.wrapAll._
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer

@Component
class ControlTaskService {

  def prepareControlTask(ci: ConfigurationItem, controlTaskName: String, params: Parameters, workDir: WorkDir): TaskSpecification = {
    val method: MethodDescriptor = DescriptorRegistry.getDescriptor(ci.getType).getControlTask(controlTaskName)
    checkArgument(method != null, "ConfigurationItem %s of type %s does not have a control task named %s.", ci.getId, ci.getType, controlTaskName)

    val steps: util.List[Step] = method.invoke(ci, params)

    val desc: String = descriptionFor(ci, controlTaskName)
    val plan: PhasedPlan = asPhasedPlan(controlTaskName, buildExecutionPlan(ci, controlTaskName, steps.toList))
    val planForSatellites: PhasedPlan = Satellites.prepareForSatelliteExecution(plan)
    val phaseContainer = Plans.toBlockBuilder(planForSatellites).build()
    val taskSpec = new TaskSpecification(desc, Permissions.getAuthentication, workDir, phaseContainer)
    taskSpec.getListeners.add(new WorkdirCleanerTrigger(workDir))

    taskSpec.getMetadata.putAll(Map(
      TASK_TYPE -> TaskType.CONTROL.name,
      TASK_NAME -> controlTaskName,
      CONTROL_TASK_TARGET_CI -> ci.getId))

    // cloud operations are identified by having a control task name instantiate or destroy
    // ignore instantiation/destroy of single host templates
    controlTaskName match {
      case NodeNames.DESTROY_CONTROL_TASK_NAME if instanceOf(ci, "cloud.Environment") =>
        taskSpec.getMetadata.putAll(Map(
          CLOUD_OPERATION -> controlTaskName,
          CLOUD_ENVIRONMENT_ID -> ci.getId,
          CLOUD_ENVIRONMENT_TEMPLATE_ID -> ci.getProperty[BaseConfigurationItem]("template").getId
        ))
      case NodeNames.INSTANTIATE_CONTROL_TASK_NAME if params != null && instanceOf(params, "cloud.CloudEnvironmentParameters") =>
        taskSpec.getMetadata.putAll(Map(
          CLOUD_OPERATION -> controlTaskName,
          CLOUD_ENVIRONMENT_TEMPLATE_ID -> ci.getId,
          CLOUD_ENVIRONMENT_ID -> params.getProperty[String](ControlTaskService.CLOUD_PARAMS_ENVIRONMENT_ID)
        ))
      case _ =>
    }

    taskSpec
  }

  private[controltask] def buildExecutionPlan(ci: ConfigurationItem, controlTaskName: String, steps: List[Step]): ExecutablePlan = {
    steps match {
      case Nil => throw new IllegalStateException(s"Control task $controlTaskName on ${ci.getId} resulted in no steps.")
      case head :: tail =>
        val aggregator = (satelliteOfStep(head), ArrayBuffer(head)) :: Nil
        val folded: List[(Option[Satellite], mutable.Buffer[Step])] = steps.tail.foldLeft(aggregator)({
          case (acc, step) if acc.head._1 == satelliteOfStep(step) =>
            acc.head._2.append(step)
            acc
          case (acc, step) => (satelliteOfStep(step), ArrayBuffer(step)) :: acc
        })
        val builders: List[StepPlan] = folded.map({ case (sat, ss) => new StepPlan(descriptionFor(ci, controlTaskName, sat), ss.toList, List(), sat.orNull) }).reverse
        builders match {
          case hd :: Nil => hd
          case xs => new SerialPlan(descriptionFor(ci, controlTaskName), xs, null)
        }
    }
  }

  private[this] def satelliteOfStep(step: Step): Option[Satellite] = Option(step).collect({case sa: SatelliteAware if sa.getSatellite != null => sa.getSatellite})

  private[this] def asPhasedPlan(controlTaskName: String, plan: ExecutablePlan): PhasedPlan = {
    val phase: PlanPhase = new PlanPhase(plan, s"Execute the $controlTaskName control task", List(), false)
    new PhasedPlan(List(phase), List())
  }

  private[this] def instanceOf(ci: ConfigurationItem, t: String) = ci.getType.instanceOf(Type.valueOf(t))

  private[this] def descriptionFor(ci: ConfigurationItem, controlTaskName: String, satellite: Option[Satellite]): String = {
    satellite match {
      case Some(x) => descriptionFor(ci, controlTaskName) + s" (executed on $x)"
      case None => descriptionFor(ci, controlTaskName)
    }
  }

  private[this] def descriptionFor(ci: ConfigurationItem, controlTaskName: String) = s"Control task [$controlTaskName] for ${ci.getId}"
}

object ControlTaskService {
  val CLOUD_PARAMS_ENVIRONMENT_ID: String = "environmentId"
}
