package com.xebialabs.deployit.service.deployment;

import com.xebialabs.deployit.core.MapStringStringView;
import com.xebialabs.deployit.core.StringValue;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployed;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployedContainer;
import com.xebialabs.deployit.plugin.api.udm.artifact.EmbeddedDeployedArtifact;
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

/**
 * Resolves the 'placeholders' field of a DeployedArtifact.
 */
@Component
public class DeployedArtifactPlaceholdersResolver {

    public static final String PLACEHOLDERS_FIELD = "placeholders";

    @SuppressWarnings("unchecked")
    public void resolvePlaceholders(EmbeddedDeployedContainer<?, ?> deployed, ConsolidatedDictionary dictionary, Set<String> missingPlaceholdersAggregator) {
        resolveDeployedPlaceholders(deployed, dictionary, missingPlaceholdersAggregator);
        resolveEmbeddedArtifactsPlaceholders(deployed, dictionary, missingPlaceholdersAggregator);
    }

    private void resolveEmbeddedArtifactsPlaceholders(final EmbeddedDeployedContainer<?, ?> deployed, final ConsolidatedDictionary dictionary, Set<String> missingPlaceholdersAggregator) {
        deployed.getType().getDescriptor().getPropertyDescriptors().stream().filter(this::isEmbeddedArtifactProperty).forEach(pd -> {
            Collection<EmbeddedDeployed> embeddedDeployeds = deployed.getProperty(pd.getName());
            embeddedDeployeds.forEach(embeddedDeployed -> resolvePlaceholders(embeddedDeployed, dictionary, missingPlaceholdersAggregator));
        });
    }

    private void resolveDeployedPlaceholders(EmbeddedDeployedContainer<?, ?> deployed, ConsolidatedDictionary dictionary, Set<String> missingPlaceholdersAggregator) {
        if (!deployed.hasProperty(PLACEHOLDERS_FIELD)) {
            return;
        }

        Set<String> placeholders = deployed.getDeployable().getProperty(PLACEHOLDERS_FIELD);

        deployed.setProperty(PLACEHOLDERS_FIELD, resolvePlaceholders(placeholders, dictionary, missingPlaceholdersAggregator));
    }

    private Map<String, String> resolvePlaceholders(Set<String> deployablePlaceholders, ConsolidatedDictionary dictionary, Set<String> missingPlaceholdersAggregator) {
        MapStringStringView newDeployedPlaceholders = new MapStringStringView();

        for (String deployablePlaceholder : deployablePlaceholders) {
            StringValue placeholderValue = dictionary.resolveScannedPlaceholder(deployablePlaceholder);
            if (placeholderValue == null) {
                placeholderValue = new StringValue("");
                missingPlaceholdersAggregator.add(deployablePlaceholder);
            }
            newDeployedPlaceholders.put(deployablePlaceholder, placeholderValue);
        }
        return newDeployedPlaceholders;
    }

    private boolean isEmbeddedArtifactProperty(PropertyDescriptor propertyDescriptor) {
        return propertyDescriptor.isAsContainment() &&
                (propertyDescriptor.getKind() == PropertyKind.SET_OF_CI || propertyDescriptor.getKind() == PropertyKind.LIST_OF_CI) &&
                propertyDescriptor.getReferencedType().isSubTypeOf(Type.valueOf(EmbeddedDeployedArtifact.class));
    }

}
