package com.xebialabs.deployit.service.deployment;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.xld.AppliedDistribution;
import com.xebialabs.deployit.service.replacement.Dictionaries;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;

import static java.util.stream.Collectors.groupingBy;

@Component
public class DeployedProcessorsFactory {

    private final DeploymentContext.BeanSupplier beanSupplier;
    private final TypeCalculator calculator;
    private final Map<Type, Class<? extends DeployedGenerator>> deployableSpecificGenerators = new HashMap<>();

    @Autowired
    public DeployedProcessorsFactory(DeployedArtifactPlaceholdersResolver deployedArtifactPlaceholdersResolver, DeployedPropertySetter deployedPropertySetter, TypeCalculator calculator) {
        final Map<Class<?>, Object> beans = new HashMap<>();
        beans.put(deployedArtifactPlaceholdersResolver.getClass(), deployedArtifactPlaceholdersResolver);
        beans.put(deployedPropertySetter.getClass(), deployedPropertySetter);
        beans.put(calculator.getClass(), calculator);
        beanSupplier = new DeploymentContext.BeanSupplier() {
            @Override
            public <T> T getBean(Class<T> beanClass) {
                return beanClass.cast(beans.get(beanClass));
            }
        };
        this.calculator = calculator;
    }

    public void registerDeployedGenerator(Type deployableType, Class<? extends DeployedGenerator> deployedGenerator) {
        deployableSpecificGenerators.put(deployableType, deployedGenerator);
    }

    public DeployedGenerator createTagDeployedGenerator() {
        return new TagMatchingDeployedGenerator(createCoreDeployedGenerator());
    }

    public DeployedGenerator createCoreDeployedGenerator() {
        return perTypeDeployedGenerator(
                new PlaceholderPropertyDeployedGenerator(
                        new PropertyResolverDeployedGenerator(
                                new TypeFilterDeployedGenerator(
                                        new BasicDeployedGenerator()
                                )
                        )
                )
        );
    }

    public DeployedGenerator createUpgradeDeployedGenerator() {
        return perTypeDeployedGenerator(
                new UpdateSameDeployableChainGenerator(
                        new PlaceholderPropertyDeployedGenerator(
                                new PropertyResolverDeployedGenerator(
                                        new BasicDeployedGenerator()
                                )
                        )
                )
        );
    }

    private PerDeployableTypeGenerator perTypeDeployedGenerator(DeployedGenerator generator) {
        return new PerDeployableTypeGenerator(deployableSpecificGenerators, generator);
    }

    public DeploymentContext createContextForSpecificType(AppliedDistribution appliedDistribution, Dictionaries dictionaries, Type deployedType) {
        return new DeploymentContext(appliedDistribution, dictionaries, beanSupplier, (deployable, container) -> deployedType, deployable -> Collections.emptyList());
    }

    public DeploymentContext createContextWithCalculatedType(AppliedDistribution appliedDistribution, Dictionaries dictionaries) {
        return new DeploymentContext(appliedDistribution, dictionaries, beanSupplier,
                (deployable, container) -> calculator.findMostSpecificDeployedType(deployable.getType(), container.getType()),
                deployable -> Collections.emptyList());
    }

    public DeploymentContext createContextForUpgrade(AppliedDistribution appliedDistribution, Dictionaries dictionaries, Set<Deployed> previousDeployeds) {
        Map<String, List<Deployed>> deployedsPerDeployableId = previousDeployeds.stream().filter(x -> x.getDeployable() != null).collect(groupingBy(deployed -> deployed.getDeployable().getName()));
        return new DeploymentContext(appliedDistribution, dictionaries, beanSupplier,
                (deployable, container) -> calculator.findMostSpecificDeployedType(deployable.getType(), container.getType()),
                deployable -> deployedsPerDeployableId.get(deployable.getName())
        );
    }
}
