package com.xebialabs.deployit.service.deployment;

import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployable;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployed;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployedContainer;
import com.xebialabs.deployit.plugin.api.xld.AppliedDistribution;
import com.xebialabs.deployit.server.api.util.IdGenerator;
import com.xebialabs.deployit.service.deployment.setter.AbstractPropertySetter;
import com.xebialabs.deployit.service.deployment.setter.DefaultPropertySetter;
import com.xebialabs.deployit.service.deployment.setter.FallbackPropertySetter;
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary;
import com.xebialabs.deployit.service.replacement.DictionaryValueException;
import com.xebialabs.deployit.service.replacement.MustachePlaceholderScanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import static com.xebialabs.deployit.checks.Checks.checkArgument;

@Component
public class DeployedPropertySetter {

    private final TypeCalculator calculator;

    private final ServerConfiguration serverConfiguration;

    private final DefaultPropertySetter defaultPropertySetter;

    private final FallbackPropertySetter fallbackPropertySetter;

    @Autowired
    public DeployedPropertySetter(TypeCalculator typeCalculator, ServerConfiguration serverConfiguration,
                                  DefaultPropertySetter defaultPropertySetter, FallbackPropertySetter fallbackPropertySetter) {
        this.calculator = typeCalculator;
        this.serverConfiguration = serverConfiguration;
        this.defaultPropertySetter = defaultPropertySetter;
        this.fallbackPropertySetter = fallbackPropertySetter;
    }

    public void setProperties(EmbeddedDeployedContainer<?, ?> deployed, EmbeddedDeployedContainer<?, ?> previousDeployed, ConsolidatedDictionary dictionary, Set<String> missingPlaceholdersAggregator) {
        checkArgument(previousDeployed == null || deployed.getType().equals(previousDeployed.getType()), "New and existing target types should be equal when upgrading (%s)", deployed.getType());
        for (PropertyDescriptor deployedPropDesc : deployed.getType().getDescriptor().getPropertyDescriptors()) {

            if (deployedPropDesc.isDeployedSpecific()) {
                continue;
            }

            if (isEmbeddedProperty(deployedPropDesc)) {
                generateEmbeddeds(deployed, deployedPropDesc, dictionary, missingPlaceholdersAggregator);
                continue;
            }

            final ConfigurationItem currentDeployable = deployed.getDeployable();
            ConfigurationItem previousDeployable = previousDeployed == null ? null : previousDeployed.getDeployable();
            setProperty(deployed, previousDeployed, currentDeployable, previousDeployable, dictionary, deployedPropDesc, missingPlaceholdersAggregator);
        }
    }

    public void setProperties(AppliedDistribution currentDeployedApp, AppliedDistribution existingDeployedApp, ConsolidatedDictionary dictionary, Set<String> missingPlaceholdersAggregator) {
        final ConfigurationItem previousDeployable = existingDeployedApp == null ? null : existingDeployedApp.getVersion();
        final ConfigurationItem currentDeployable = currentDeployedApp.getVersion();

        for (PropertyDescriptor deployedPropDesc : currentDeployedApp.getType().getDescriptor().getPropertyDescriptors()) {

            if (deployedPropDesc.isDeployedSpecific()) {
                continue;
            }
            setProperty(currentDeployedApp, existingDeployedApp, currentDeployable, previousDeployable, dictionary, deployedPropDesc, missingPlaceholdersAggregator);
        }
        for (PropertyDescriptor pd : currentDeployable.getType().getDescriptor().getPropertyDescriptors()) {
            if (currentDeployedApp.hasProperty(pd.getName())) {
                PropertyDescriptor deployedAppPropertyDescriptor = currentDeployedApp.getType().getDescriptor().getPropertyDescriptor(pd.getName());
                setProperty(currentDeployedApp, existingDeployedApp, currentDeployable, previousDeployable, dictionary, deployedAppPropertyDescriptor, missingPlaceholdersAggregator);
            }
        }

    }

    private void setProperty(ConfigurationItem currentTarget, ConfigurationItem previousTarget, ConfigurationItem currentSource,
                             ConfigurationItem previousSource, ConsolidatedDictionary dictionary, PropertyDescriptor targetPropertyDescriptor, Set<String> missingPlaceholdersAggregator) {

        logger.debug("Going to set property {} of currentTarget {}", targetPropertyDescriptor.getFqn(), currentTarget);

        AbstractPropertySetter propertySetter = serverConfiguration.isServerMappingOverrideDeployedFieldsOnUpdate() ?
                defaultPropertySetter : fallbackPropertySetter;
        propertySetter.setProperty(currentTarget, previousTarget, currentSource, previousSource, dictionary, targetPropertyDescriptor, missingPlaceholdersAggregator);
    }

    private void generateEmbeddeds(EmbeddedDeployedContainer<?, ?> deployed, final PropertyDescriptor deployedPropDesc, ConsolidatedDictionary dictionary, Set<String> missingPlaceholdersAggregator) {
        final ConfigurationItem deployable = deployed.getDeployable();
        String name = deployedPropDesc.getName();
        Collection<EmbeddedDeployed<?, ?>> embeddedDeployeds = deployedPropDesc.getKind() == PropertyKind.LIST_OF_CI ? Lists.<EmbeddedDeployed<?, ?>>newArrayList() : Sets.<EmbeddedDeployed<?, ?>>newHashSet();
        PropertyDescriptor deployableProperty = deployable.getType().getDescriptor().getPropertyDescriptor(name);
        if (deployableProperty == null) {
            logger.info("Deployed [{}] has embedded deployeds in property [{}] but source [{}] has no such property.", deployed.getId(), deployedPropDesc.getFqn(), deployable.getId());
            return;
        }
        @SuppressWarnings("unchecked")
        Collection<EmbeddedDeployable> embeddedDeployables = (Collection<EmbeddedDeployable>) deployableProperty.get(deployable);

        for (EmbeddedDeployable embeddedDeployable : embeddedDeployables) {
            final List<Type> embeddedDeployedTypes = calculator.findAllMostSpecificDeployedTypesForDeployableAndContainerTypes(embeddedDeployable.getType(), deployed.getType());
            Collection<Type> filtered = Collections2.filter(embeddedDeployedTypes, input -> input.instanceOf(deployedPropDesc.getReferencedType()));
            if (filtered.isEmpty()) {
                logger.info("Not found a matching EmbeddedDeployed type for [{}] and [{}]", embeddedDeployable.getType(), deployed.getType());
            } else if (filtered.size() > 1) {
                logger.error("Found more than 1 ({}) compatible EmbeddedDeployed type for [{}] and [{}]", filtered, embeddedDeployable.getType(), deployed.getType());
                throw new Checks.IncorrectArgumentException("Found more than 1 (%s) compatible EmbeddedDeployed type for [%s] and [%s]", filtered, embeddedDeployable.getType(), deployed.getType());
            } else {
                Type embeddedDeployedType = filtered.iterator().next();
                EmbeddedDeployed embeddedDeployed = embeddedDeployedType.getDescriptor().newInstance(IdGenerator.generateId(deployed, embeddedDeployable.getId()));
                //noinspection unchecked
                embeddedDeployed.setDeployable(embeddedDeployable);
                //noinspection unchecked
                embeddedDeployed.setContainer(deployed);

                if (MustachePlaceholderScanner.hasPlaceholders(embeddedDeployed.getName())) {
                    try {
                        dictionary.resolveDeployedName(embeddedDeployed);
                    } catch (DictionaryValueException e) {
                        throw new DeployedApplicationFactory.IncorrectDeployedException(e, "Couldn't generate name for target from [%s]", embeddedDeployable.getId());
                    }
                }
                embeddedDeployeds.add(embeddedDeployed);
                setProperties(embeddedDeployed, null, dictionary, missingPlaceholdersAggregator);
            }
        }

        deployedPropDesc.set(deployed, embeddedDeployeds);
    }

    private boolean isEmbeddedProperty(PropertyDescriptor propertyDescriptor) {
        return propertyDescriptor.isAsContainment() &&
                (propertyDescriptor.getKind() == PropertyKind.SET_OF_CI || propertyDescriptor.getKind() == PropertyKind.LIST_OF_CI) &&
                propertyDescriptor.getReferencedType().isSubTypeOf(Type.valueOf(EmbeddedDeployed.class));
    }

    private static final Logger logger = LoggerFactory.getLogger(DeployedPropertySetter.class);
}
