package com.xebialabs.deployit.service.importer;

import com.google.common.base.Predicate;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.server.api.importer.ImportSource;
import com.xebialabs.deployit.util.TFiles;

import de.schlichtherle.truezip.file.TFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import java.io.File;
import java.util.Set;

/**
 */
class PackageScanner {

    static Set<String> scan(File directory, Predicate<File> isPackage) {
        if (!directory.exists() || !directory.isDirectory()) {
            throw new IllegalArgumentException("Directory " + directory.getAbsolutePath() + " does not exist or is not a directory");
        }
        
        Set<String> result = Sets.newHashSet();
        logger.debug("Listing importable packages in {}", directory.getAbsolutePath());

        scanDirectory(directory, result, "", isPackage);

        logger.debug("Returning the following packages: {}", result );

        return result;

    }

    private static void scanDirectory(final File directory, final Set<String> result, String pathPrefix, Predicate<File> isPackage) {
        final File[] files = directory.listFiles();
        logger.debug("Found {} possible packages.", files.length);

        for (File f : files) {
            if (isPackage.apply(f)) {
                result.add(pathPrefix + f.getName());
            } else if (f.isDirectory()) {
                scanDirectory(f, result, pathPrefix + f.getName() + "/", isPackage);
            }
        }
    }


    static boolean isDarPackage(final File f) {
        return f.isFile() && StringUtils.endsWithIgnoreCase(f.getName(), ".dar");
    }

    static boolean isXmlManifestPackage(final File f) {
        return (isDarPackage(f) || f.isDirectory()) && containsXmlManifest(f);
    }

    private static boolean containsXmlManifest(File f) {
        TFile tFile = new TFile(f);
        try {
            return new TFile(tFile, "deployit-manifest.xml").exists();
        } finally {
            TFiles.umountQuietly(tFile);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(PackageScanner.class);
}
