package com.xebialabs.deployit.service.importer.reader;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.google.common.annotations.VisibleForTesting;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.input.SAXBuilder;

import com.google.common.base.Function;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.server.api.importer.ImportSource;
import com.xebialabs.deployit.server.api.importer.ImportingContext;
import com.xebialabs.deployit.server.api.importer.PackageInfo;
import com.xebialabs.deployit.service.importer.ImporterException;
import com.xebialabs.deployit.util.TFiles;

import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.file.TFileInputStream;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.transform;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.booter.local.utils.Strings.isBlank;

public class XmlManifestReader implements ManifestReader {
    private Element element;

    @VisibleForTesting
    public XmlManifestReader() {

    }
    public XmlManifestReader(ImportSource source) {
        File packageFile = source.getFile();
        element = readXmlManifest(packageFile);
    }

    private Element readXmlManifest(File packageFile) {
        TFile manifest = new TFile(packageFile, "deployit-manifest.xml");
        try(TFileInputStream is = new TFileInputStream(manifest)) {
            return buildManifestRoot(is);
        } catch (IOException e) {
            throw new ImporterException(e, "Could not read manifest");
        } finally {
            TFiles.umountQuietly(manifest);
            TFiles.umountQuietly(new TFile(packageFile));
        }
    }

    Element buildManifestRoot(InputStream is) throws IOException{
        try {
            SAXBuilder saxBuilder = new SAXBuilder();
            saxBuilder.setEntityResolver(new EntityResolver() {
                @Override
                public InputSource resolveEntity(String publicId, String systemId) throws SAXException, IOException {
                    throw new IllegalArgumentException("External Entity is not allowed in the manifest file");
                }
            });
            return saxBuilder.build(is).getRootElement();
        } catch (JDOMException e) {
            throw new ImporterException(e, "Could not read manifest file, invalid xml.");
        }
    }

    @Override
    public void readPackageData(PackageInfo packageInfo, ImportingContext context) {
        check(!isBlank(element.getAttributeValue("application")), "Missing or empty required attribute 'application' in XML Manifest for [%s]", packageInfo.getSource().getFile().getName());
        check(!isBlank(element.getAttributeValue("version")), "Missing or empty required attribute 'version' in XML Manifest for [%s]", packageInfo.getSource().getFile().getName());
        packageInfo.setApplicationName(element.getAttributeValue("application"));
        packageInfo.setApplicationVersion(element.getAttributeValue("version"));
        packageInfo.setApplicationRoot(versionReader().distributionType().getDescriptor().getRootName());
    }

    private static void check(boolean bool, String template, Object... args) {
        if (!bool) {
            throw new ImporterException(template, args);
        }
    }

    @Override
    public ManifestCiReader versionReader() {
        return new XmlManifestCiReader(element);
    }

    public static class XmlManifestCiReader implements ManifestCiReader {
        private Element element;

        public XmlManifestCiReader(Element element) {
            this.element = element;
        }

        @Override
        public Type type() {
            return Type.valueOf(element.getName());
        }

        @Override
        public Type distributionType() {
            return getDistributionPropertyDescriptor().getReferencedType();
        }

        private PropertyDescriptor getDistributionPropertyDescriptor() {
            Descriptor versionDescriptor = type().getDescriptor();
            Optional<PropertyDescriptor> parentHolder = versionDescriptor.getPropertyDescriptors().stream().filter(pd -> pd.getKind() == PropertyKind.CI && pd.isAsContainment()).findFirst();
            return parentHolder.orElseThrow(() -> new IllegalStateException("Cannot find distribution type"));
        }

        @Override
        public String name() {
            if (isBlank(element.getAttributeValue("name"))) {
                throw new ImporterException("Missing or empty required attribute 'name' for CI '%s' in XML Manifest", element.getName());
            }
            return element.getAttributeValue("name");
        }

        @Override
        public String file() {
            return element.getAttributeValue("file");
        }

        @Override
        public boolean hasProperty(String name) {
            return element.getChild(name) != null;
        }

        public List<String> allProperties() {
            return newArrayList(transform(element.getChildren(), new Function<Element, String>() {
                public String apply(org.jdom2.Element input) {
                    return input.getName();
                }
            }));
        }

        @Override
        public String propertyAsString(String name) {
            checkOneElement(name);
            return element.getChildTextTrim(name);
        }

        private void checkOneElement(String name) {
            check(element.getChildren(name).size() == 1, "More than 1 element named '%s' found for CI '%s[%s]' in XML Manifest", name, element.getAttributeValue("name"), element.getName());
        }

        @Override
        public String propertyAsCiRef(String name) {
            checkOneElement(name);
            return element.getChild(name).getAttributeValue("ref");
        }

        @Override
        public List<String> propertyAsStringCollection(String name) {
            checkOneElement(name);
            List<String> strings = newArrayList();
            for (Element value : element.getChild(name).getChildren("value")) {
                strings.add(value.getTextTrim());
            }
            // backwards compatibility for properties that change from single to multiple values (like orchestrator)
            String textValue = element.getChild(name).getTextTrim();
            if(strings.isEmpty() && !textValue.isEmpty()){
                strings.add(textValue);
            }
            return strings;
        }

        @Override
        public List<String> propertyAsCiRefCollection(String name) {
            checkOneElement(name);
            List<String> refs = newArrayList();
            for (Element ci : element.getChild(name).getChildren("ci")) {
                refs.add(ci.getAttributeValue("ref"));
            }
            return refs;
        }

        @Override
        public Map<String, String> propertyAsMapStringString(String name) {
            checkOneElement(name);
            HashMap<String,String> map = newHashMap();
            for (Element entry : element.getChild(name).getChildren("entry")) {
                map.put(entry.getAttributeValue("key"), entry.getTextTrim());
            }
            return map;
        }

        @Override
        public List<ManifestCiReader> propertyAsNestedCis(String name) {
            checkOneElement(name);
            List<Element> children = element.getChild(name).getChildren();
            return newArrayList(transform(children, new Function<Element, ManifestCiReader>() {
                public ManifestCiReader apply(org.jdom2.Element input) {
                    return new XmlManifestCiReader(input);
                }
            }));
        }
    }
}
