package com.xebialabs.deployit.core.upgrade

import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, Type}
import com.xebialabs.deployit.server.api.repository.RawRepository
import com.xebialabs.deployit.server.api.upgrade.{UpgradeException, Upgrade, Version}
import org.slf4j.{Logger, LoggerFactory}

import scala.collection.JavaConversions._
import javax.jcr.{Node, Property}
import com.xebialabs.deployit.repository.JcrPathHelper

object Deployit50OrchestratorConversion {

  def wrap(str: String) = s"u{$str}"

  def unwrap(str: String) = {
    val Pattern = """u\{(.*)\}""".r
    str match {
      case Pattern(orchName) => orchName
      case s => s
    }
  }

}

class Deployit50OrchestratorConversion extends Upgrade {
  import Deployit50OrchestratorConversion._

  private val logger: Logger = LoggerFactory.getLogger(classOf[Deployit50OrchestratorConversion])

  private val orchestratorsProperty: String = "orchestrator"

  private val conversions =
    Map("container-by-container-serial"        -> wrap("sequential-by-container"),
        "composite-package"                    -> wrap("sequential-by-composite-package"),
        "group-based"                          -> wrap("sequential-by-deployment-group"),
        wrap("container-by-container-serial") -> wrap("sequential-by-container"),
        wrap("composite-package")             -> wrap("sequential-by-composite-package"),
        wrap("group-based")                   -> wrap("sequential-by-deployment-group"))

  def doUpgrade(repository: RawRepository): Boolean = {

    val subVersionNodes = DescriptorRegistry.getSubtypes(Type.valueOf("udm.Version")).flatMap(repository.findNodesByType(_).toList)
    val versionNodes = repository.findNodesByType(Type.valueOf("udm.Version")).toList
    val applicationNodes = repository.findNodesByType(Type.valueOf("udm.DeployedApplication")).toList

    try {
      convertNodes(versionNodes ++ subVersionNodes ++ applicationNodes)
    } catch {
      case e: Exception => throw new UpgradeException(e.getMessage, e)
    }

    true
  }

  private def convertNodes(nodes: List[Node]) = {
    nodes.foreach {
      node =>
        if (node.hasProperty(orchestratorsProperty)) {
          val property = node.getProperty(orchestratorsProperty)

          /*
           * This method needs to deal with orchestrators writen as Single value and Multi value properties,
           * Because the KIND of the orchestrator property has been changed earlier from STRING to LIST_OF_STRING
           */
          val values: Array[String] = property.isMultiple match {
            case true => property.getValues.map(_.getString)
            case false => Array(property.getValue.getString)
          }

          val newValues = values.filterNot(v => v.equals("default") || v.equals(wrap("default")) ).map(s => conversions.getOrElse(s, s))
          updateProperty(values, newValues, property, node)
        }
    }
  }


  private def updateProperty(values: Array[String], newValues: Array[String], property: Property, node: Node) {
    if (!values.sameElements(newValues) || !property.isMultiple) {
      // if property is single-value we need to remove it so we can create a new multi-value property
      if (!property.isMultiple) {
        property.remove()
      }
      node.setProperty(orchestratorsProperty, newValues)
      infoMessage(node, values, newValues)
    }
  }

  private def infoMessage(node: Node, values: Array[String], newValues: Array[String]) = {
    val ciId = JcrPathHelper.getIdFromAbsolutePath(node.getPath)

    if (node.getProperty(orchestratorsProperty).isMultiple) {
      logger.info(s"Upgrader: Converting orchestrators of CI with id $ciId from: ${values.toList.map(unwrap)} to: ${newValues.toList.map(unwrap)}")
    } else {
      logger.info(s"Upgrader: Updating $ciId property kind string to list_of_string : ${newValues.toList.map(unwrap)}")
    }
  }

  def upgradeVersion = Version.valueOf("deployit", "5.0")

}