package com.xebialabs.deployit.core.upgrade

import java.io.{InputStream, IOException}
import java.net.URLEncoder
import java.security.{DigestInputStream, MessageDigest}
import javax.jcr.{RepositoryException, Node}

import com.google.common.base.Charsets
import com.google.common.io.{ByteSource, ByteStreams}
import com.xebialabs.deployit.io.Exploder
import com.xebialabs.deployit.jcr.JcrConstants
import com.xebialabs.deployit.jcr.JcrConstants.FILENAME_PROPERTY_NAME
import com.xebialabs.deployit.plugin.api.reflect.{Type, DescriptorRegistry}
import com.xebialabs.deployit.plugin.api.udm.artifact.FolderArtifact
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact.{CHECKSUM_PROPERTY_NAME, FILE_URI_PROPERTY_NAME}
import com.xebialabs.deployit.repository.JcrPathHelper
import com.xebialabs.deployit.server.api.repository.RawRepository
import com.xebialabs.deployit.server.api.upgrade.{UpgradeException, Version, Upgrade}
import com.xebialabs.deployit.util.{DevNull, JavaCryptoUtils}
import com.xebialabs.xlplatform.utils.ResourceManagement.using
import grizzled.slf4j.Logging
import collection.convert.wrapAll._
import scala.util.{Success, Failure, Try}

/**
 * Ensures that all the [[com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact]]s have `checksum` and `fileUri` properties set.
 */
class Deployit500ArtifactUpgrader extends Upgrade with Logging {

  override def doUpgrade(repository: RawRepository): Boolean = Try {
    DescriptorRegistry.getSubtypes(Type.valueOf("udm.SourceArtifact")).foreach { t =>
      logger.info(s"Searching all nodes of type $t")
      repository.findNodesByType(t).foreach { node =>
        val ciId = JcrPathHelper.getIdFromAbsolutePath(node.getPath)
        if (!node.hasProperty(FILE_URI_PROPERTY_NAME)) {
          logger.info(s"Updating $FILE_URI_PROPERTY_NAME property of $ciId")
          val filename: String = URLEncoder.encode(node.getProperty(FILENAME_PROPERTY_NAME).getString, Charsets.UTF_8.name())
          node.setProperty(FILE_URI_PROPERTY_NAME, s"jcr:$filename")
        }

        if (!node.hasProperty(CHECKSUM_PROPERTY_NAME)) {
          logger.info(s"Updating $CHECKSUM_PROPERTY_NAME property of $ciId")
          val checksum = calculateChecksum(t, node)
          node.setProperty(CHECKSUM_PROPERTY_NAME, checksum)
        }
      }
    }
  } match {
    case Failure(e) => throw new UpgradeException(s"Failed to run ${this.getClass.getSimpleName}.", e.asInstanceOf[Exception])
    case Success(_) => true
  }


  def upgradeVersion = Version.valueOf("deployit", "5.0")

  private def calculateChecksum(`type`: Type, node: Node): String = {
    val sha1: MessageDigest = JavaCryptoUtils.getSha1
    val inputSupplier = getDataInputSupplier(node)
    if (`type`.isSubTypeOf(Type.valueOf(classOf[FolderArtifact]))) {
      Exploder.calculateCheckSum(inputSupplier, sha1)
    } else {
      using(inputSupplier.openStream) { input =>
        ByteStreams.copy(new DigestInputStream(input, sha1), new DevNull)
      }
    }
    JavaCryptoUtils.digest(sha1)
  }

  private def getDataInputSupplier(node: Node) = new ByteSource {
    def openStream: InputStream = {
      try {
        node.getProperty(JcrConstants.DATA_PROPERTY_NAME).getBinary.getStream
      } catch {
        case e: RepositoryException => throw new IOException(e)
      }
    }
  }
}
