package com.xebialabs.xlplatform.rest.script.endpoints

import spray.routing.{RejectionHandler, AuthenticationFailedRejection, Route, HttpService}
import com.xebialabs.xlplatform.rest.script.jython.{ScriptDone, JythonScriptExecutorActor, RunScript}
import akka.pattern.{AskTimeoutException, ask}
import akka.util.Timeout
import scala.concurrent.duration._
import spray.http.{StatusCode, StatusCodes}
import com.xebialabs.xlplatform.rest.script.ui._
import scala.util.{Success, Failure}

import scala.Some

import com.xebialabs.xlplatform.rest.script.endpoints.json.{ScriptResponse, ScriptResponseProtocol}
import spray.httpx.SprayJsonSupport
import spray.routing.authentication.{BasicAuth, UserPass}
import scala.concurrent.Future
import com.xebialabs.xldeploy.rest.SpringContextHolder
import org.springframework.security.authentication.{UsernamePasswordAuthenticationToken, AuthenticationManager}
import spray.routing.AuthenticationFailedRejection.{CredentialsRejected, CredentialsMissing}
import akka.actor.{ActorRef, ActorLogging, Props, Actor}
import spray.http.HttpHeaders.RawHeader

trait ScriptExtensionRoute extends HttpService with SprayJsonSupport with ScriptResponseProtocol with MenuXmlProtocol {

  import scala.concurrent.ExecutionContext.Implicits.global
  implicit val timeout = Timeout(15 seconds)

  case object GetMenus

  lazy val menuActor = actorRefFactory.actorOf(Props.apply(new Actor with ActorLogging {
    override def receive: Receive = {
      case GetMenus => val menus: List[Menu] = UiMenuExtension(context.system).menus
        log.info(s"Menus = $menus")
        sender ! menus
    }
  }))

  private def jythonActor():ActorRef = actorRefFactory.actorOf(JythonScriptExecutorActor.props())

  private def createEndpointRoute(ep: Endpoint, auth: AuthenticatedData, body: Option[String] = None): Route = {
    pathPrefix("extension") {
      path(ep.getPathMatcher) {
        import scala.collection.JavaConversions._
        parameterMultiMap { params =>
          val simplifiedParams = params.map {
            case (x: String,  List("")) =>(x, null)
            case (x: String, h :: Nil) => (x, h)
            case (x, hh) => (x, seqAsJavaList(hh))
          }
          onComplete(jythonActor ? RunScript(ep.script, simplifiedParams, auth)) {
            case Success(d @ ScriptDone(_, _, _, statusCode, headers, None)) =>
              respondWithStatus(statusCode.map(StatusCode.int2StatusCode(_)).getOrElse(StatusCodes.OK)) {
                respondWithHeaders(headers.map(p => RawHeader(p._1, p._2)).toList) {
                  complete(ScriptResponse(d))
                }
              }
            case Success(d @ ScriptDone(_, _, _, _, _, Some(e))) => complete(StatusCodes.InternalServerError, ScriptResponse(d))
            case Failure(e) if e.isInstanceOf[AskTimeoutException] => complete(StatusCodes.RequestTimeout, e)
          }
        }
      }
    }
  }

  def readCustomEndpoints(endpoints: Seq[Endpoint], auth: AuthenticatedData): List[Route] = {

    endpoints.map({
      ep =>
        ep.method match {
          case "GET" => get {
            createEndpointRoute(ep, auth)
          }
          case "POST" => post {
            entity(as[String]) {
              body =>
                createEndpointRoute(ep, auth, Some(body))
            }
          }
          case "PUT" => put {
            entity(as[String]) {
              body =>
                createEndpointRoute(ep, auth, Some(body))
            }
          }
          case "DELETE" => delete {
            createEndpointRoute(ep, auth)
          }
        }
    }).toList
  }

  private val pingRoute = pathPrefix("ping") {
    get {
      complete("Pong!")
    }
  }

  private val metadataRoute = pathPrefix("metadata") {
      (get|post) {
        complete((menuActor ? GetMenus).mapTo[List[Menu]])
      }
  }

  val customRejectionHandler = RejectionHandler {
    case AuthenticationFailedRejection(CredentialsMissing, headers) :: _ => complete(StatusCodes.Unauthorized, "Please login first")
    case AuthenticationFailedRejection(CredentialsRejected, headers) :: _ => complete(StatusCodes.Forbidden, "You do not have access")
  }

  private def clientAuthenticator(userPass: Option[UserPass]): Future[Option[AuthenticatedData]] = {
    userPass match {
      case None => Future { None }
      case Some(UserPass(user, pass)) =>
        val authManager = SpringContextHolder.getApplicationContext.getBean("authenticationManager").asInstanceOf[AuthenticationManager]
        val auth = authManager.authenticate(new UsernamePasswordAuthenticationToken(user, pass))

        if (!auth.isAuthenticated) Future{None}
        else Future{Some(AuthenticatedData(auth))}
    }
  }

  def routes(e: Seq[Endpoint], authenticator: Option[UserPass] => Future[Option[AuthenticatedData]]) =
    handleRejections(customRejectionHandler) {
    pathPrefix("api") {
      authenticate(BasicAuth(authenticator, realm = "XLDeploy extension API")) { auth: AuthenticatedData =>
        readCustomEndpoints(e, auth).foldLeft(metadataRoute)(_ ~ _) ~ pingRoute
      }
    }
  }

  def routes(e: Seq[Endpoint]): Route = routes(e, clientAuthenticator)

}
