package com.xebialabs.xlplatform.rest.script.jython

import akka.actor._
import java.util.{Map => JMap}
import javax.script.{ScriptEngineManager, ScriptContext, SimpleScriptContext, ScriptEngine}
import spray.json.DefaultJsonProtocol
import java.io._
import scala.util.Try
import org.python.core.PySyntaxError
import com.xebialabs.xldeploy.rest.ServiceHolder
import com.xebialabs.xlplatform.rest.script.endpoints.AuthenticatedData
import org.springframework.security.core.context.SecurityContextHolder
import scala.collection.convert.wrapAll._
import com.xebialabs.xldeploy.rest.script.jython.JythonResponse
import java.net.URL
import com.xebialabs.deployit.booter.local.utils.Closeables
import JythonScriptExecutorActor.SCRIPT_ENGINE
import scala.util.Failure
import scala.Some
import scala.util.Success
import akka.event.LoggingAdapter

object JythonScriptExecutorActor {
  def props() = Props(classOf[JythonScriptExecutorActor])

  val SCRIPT_ENGINE =  new ScriptEngineManager().getEngineByName("python")
}

case class RunScript(scriptFileUrl: URL, input: Map[String, Any], auth: AuthenticatedData)

case class ScriptDone (entity: Any, stdout: String, stderr: String, statusCode: Option[Integer] = None, headers: Map[String, String] = Map(), exception: Option[Throwable] = None)

object ScriptDone {
  def apply(jr: JythonResponse, stdout: String, stderr: String, exception: Option[Throwable]): ScriptDone = {
    val entity = Option(jr.entity).map({
        case m: JMap[String, Any] => m.toMap
        case m => m
      }).getOrElse(Map())
    ScriptDone(entity, stdout, stderr, Option(jr.statusCode), jr.headers.toMap, exception)
  }
}

class InvalidScriptOutputException(msg: String) extends RuntimeException(msg)

class JythonScriptExecutorActor extends Actor with DefaultJsonProtocol with ActorLogging {

  override def receive(): Actor.Receive = {
    case RunScript(scriptUrl, input, auth) =>
      SecurityContextHolder.getContext.setAuthentication(auth.toAuthentication)
      val reader: Reader = new BufferedReader(new InputStreamReader(scriptUrl.openStream()))
      val contextHelper = ScriptContextHelper(SCRIPT_ENGINE).withAttributes(input).withClients().withEmptyResponse().withLogger(log)
      Try(SCRIPT_ENGINE.eval(reader, contextHelper.context)) match {
        case Success(_) => sender ! ScriptDone(contextHelper.jythonResponse, contextHelper.stdOut, contextHelper.stdErr, None)
        case Failure(e) if e.getCause.isInstanceOf[PySyntaxError] =>
          log.error(e.getCause, "Jython syntax error occurred.")
          sender ! ScriptDone(contextHelper.jythonResponse, contextHelper.stdOut, contextHelper.stdErr, Some(e.getCause))
        case Failure(e) =>
          log.error(e.getCause, "Jython runtime exception occurred.")
          sender ! ScriptDone(contextHelper.jythonResponse, contextHelper.stdOut, contextHelper.stdErr, Some(e.getCause))
      }

      Closeables.closeQuietly(reader)
      SecurityContextHolder.getContext.setAuthentication(null)
      self ! PoisonPill
  }
}

object ScriptContextHelper {
  def apply(se: ScriptEngine) = new ScriptContextHelper(se)
}

class ScriptContextHelper(engine: ScriptEngine) {

  final val outputParamName = "output"
  final val inputParamName = "input"
  final val responseParamName = "response"
  final val headersParamName = "headers"
  final val statusCodeParamName = "statusCode"

  val outWriter: StringWriter = new StringWriter()
  val errWriter: StringWriter = new StringWriter()

  val context = createContext

  def createContext = {
    val c = new SimpleScriptContext
    c.setWriter(new PrintWriter(outWriter))
    c.setErrorWriter(new PrintWriter(errWriter))
    c.setBindings(engine.getContext.getBindings(ScriptContext.ENGINE_SCOPE), ScriptContext.GLOBAL_SCOPE)
    c
  }


  def jythonResponse = Option(context.getAttribute(responseParamName)).map(_.asInstanceOf[JythonResponse]).getOrElse(new JythonResponse())


  def stdOut = outWriter.getBuffer.toString

  def stdErr = errWriter.getBuffer.toString

  def withAttributes(attrs: Map[String, Any]): ScriptContextHelper = {
    context.setAttribute(inputParamName, mapAsJavaMap(attrs), ScriptContext.ENGINE_SCOPE)
    this
  }

  def withEmptyResponse(): ScriptContextHelper = {
    context.setAttribute(responseParamName, new JythonResponse(), ScriptContext.ENGINE_SCOPE)
    this
  }

  def withLogger(log:LoggingAdapter): ScriptContextHelper = {
    context.setAttribute("logger", log, ScriptContext.ENGINE_SCOPE)
    this
  }

  def withClients(): ScriptContextHelper = {
    context.setAttribute("controlService", ServiceHolder.getControlService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("deploymentService", ServiceHolder.getDeploymentService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("inspectionService", ServiceHolder.getInspectionService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("metadataService", ServiceHolder.getMetadataService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("packageService", ServiceHolder.getPackageService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("permissionService", ServiceHolder.getPermissionService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("repositoryService", ServiceHolder.getRepositoryService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("roleService", ServiceHolder.getRoleService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("serverService", ServiceHolder.getServerService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("taskService", ServiceHolder.getTaskService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("taskBlockService", ServiceHolder.getTaskBlockService, ScriptContext.ENGINE_SCOPE)
    context.setAttribute("userService", ServiceHolder.getUserService, ScriptContext.ENGINE_SCOPE)
    this
  }


}
