package com.xebialabs.deployit.jetty;

import com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.IOException;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;

public class RequestHeadersEncodedAsParametersFilter implements Filter {

    public static final String X_HEADER_PREFIX = "X-Header-";

	@Override
	public void init(FilterConfig filterConfig) throws ServletException {
		logger.info("Initialized to check for Http Headers in Post Request parameters.");
	}
	
	@Override
	public void destroy() {
		//empty
	}

	@Override
	public void doFilter(ServletRequest request, ServletResponse response,
			FilterChain chain) throws IOException, ServletException {
        
        ServletRequest filteredRequest = request;
		if (request instanceof HttpServletRequest) {
			filteredRequest = processRequest((HttpServletRequest)request);
		}
		chain.doFilter(filteredRequest, response);
	}

	private ServletRequest processRequest(HttpServletRequest request) {
		return new HttpHeadersEncodedAsParametersServletRequest(request);
	}


    private static class HttpHeadersEncodedAsParametersServletRequest extends HttpServletRequestWrapper {
        HttpHeadersEncodedAsParametersServletRequest(HttpServletRequest request) {
            super(request);
        }

        @Override
        public String getHeader(String name) {
            String header = getHeaderFromParameters(name);
            if (header != null) {
                logger.debug("Header {} resolved from request parameters",name);
            } else {
                header = super.getHeader(name);
            }

            return header;
        }

        private String getHeaderFromParameters(String name) {
            String headerAsParam = X_HEADER_PREFIX + name;
            return getParameter(headerAsParam);
        }

        @Override
        public String getContentType() {
            String contentType = getHeaderFromParameters("Content-Type");
            if (contentType == null) {
                contentType = super.getContentType();
            }
            return contentType;
        }

        @Override
        public Enumeration getHeaders(String name) {
            List<String> headers = Collections.list((Enumeration<String>) super.getHeaders(name));
            String headerFromParam = getHeaderFromParameters(name);
            if (headerFromParam != null) {
                logger.debug("Header {} resolved from request parameters",name);
                headers.add(headerFromParam);
            }
            return  Collections.enumeration(headers);
        }

        @Override
        public Enumeration getHeaderNames() {
            List<String> headers = Collections.list((Enumeration<String>) super.getHeaderNames());
            Enumeration<String> params = (Enumeration<String>) getParameterNames();
            while(params.hasMoreElements()) {
                String param = params.nextElement();
                if (param.startsWith(X_HEADER_PREFIX)) {
                    String headerInParam = param.substring(X_HEADER_PREFIX.length());
                    if (!headers.contains(headerInParam)) {
                        logger.debug("Header {} resolved from request parameters", headerInParam);
                        headers.add(headerInParam);
                    }
                }
            }
            return  Collections.enumeration(headers);
        }
    }
    
	
	private static Logger logger = LoggerFactory.getLogger(RequestHeadersEncodedAsParametersFilter.class);

}
