package com.xebialabs.deployit.rest.test.api.datasets

import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.deployit.repository.RepositoryServiceHolder
import com.xebialabs.deployit.rest.test.api.DataSet
import dsl.{CiUtils, EarUtils}
import org.joda.time.DateTime

import java.util.UUID
import scala.util.Random

object DemoSet extends DataSet with CiUtils with EarUtils {

  val random = new Random()

  private val repository = RepositoryServiceHolder.getRepositoryService

  def name() = "demo"

  def generateSet(args_unused: List[String] = List()): Unit = {
    // Generate data to show off the report dashboard
    generateEnvironments()
    generateApplications()
    generateCurrentMonthData()
    generateSixMonthData()
  }

  object TaskState extends Enumeration {
    type TaskState = Value
    val SUCCESSFUL, FAILED, ABORTED = Value
  }

  private def generateEnvironments(): Unit = create { c =>
    c.directory("Environments/Ops")
    c.directory("Environments/Ops/Acc")
    c.directory("Environments/Ops/Prod")
    c.directory("Environments/Dev")

    c.environment("Ops/Acc/ACC") {
      List.empty
    }
    c.environment("Ops/Prod/PROD") {
      List.empty
    }
    c.environment("Dev/DEV-DOTNET") {
      List.empty
    }
    c.environment("Dev/DEV") {
      List.empty
    }
    c.environment("Dev/TEST") {
      List.empty
    }
  }

  private def generateApplications(): Unit = create { c =>
    createEmptyPackage(c, "PetPortal", 2)
    createEmptyPackage(c, "StockTrader", 5)
    createEmptyPackage(c, "NerdDinner", 2)
    createEmptyPackage(c, "PetClinic-ear", 2)
    createEmptyPackage(c, "PetClinic-war", 2)
  }

  import TaskState._

  private def generateCurrentMonthData(): Unit = {
    val appIds = List(
      "PetPortal",
      "StockTrader",
      "NerdDinner",
      "PetClinic-ear"
    ).map("Applications/" + _)
    val envIds = List(
      "Ops/Acc/ACC",
      "Ops/Prod/PROD",
      "Dev/DEV-DOTNET",
      "Dev/DEV",
      "Dev/TEST"
    ).map("Environments/" + _)
    val apps = readCisAsMap(appIds, repository)
    val envs = readCisAsMap(envIds, repository)
    // Strictly, we don't know whether this will all be in the current month, but at least a couple of them will be ;-)
    createCurrentTask(apps("PetPortal"), "1.0", envs("Ops/Acc/ACC"), days = 3, durationInMinutes = random.nextInt(3), SUCCESSFUL)
    createCurrentTask(apps("StockTrader"), "5.0", envs("Dev/DEV-DOTNET"), days = 3, durationInMinutes = random.nextInt(9), ABORTED)
    createCurrentTask(apps("StockTrader"), "5.0", envs("Ops/Prod/PROD"), days = 3, durationInMinutes = random.nextInt(11), SUCCESSFUL)
    createCurrentTask(apps("NerdDinner"), "2.0", envs("Dev/DEV-DOTNET"), days = 3, durationInMinutes = random.nextInt(9), SUCCESSFUL)
    createCurrentTask(apps("PetPortal"), "2.0", envs("Dev/DEV"), days = 2, durationInMinutes = random.nextInt(4), SUCCESSFUL)
    createCurrentTask(apps("StockTrader"), "5.0", envs("Dev/DEV-DOTNET"), days = 2, durationInMinutes = random.nextInt(11), SUCCESSFUL)
    createCurrentTask(apps("NerdDinner"), "2.0", envs("Ops/Acc/ACC"), days = 2, durationInMinutes = random.nextInt(11), FAILED)
    createCurrentTask(apps("NerdDinner"), "2.0", envs("Ops/Acc/ACC"), days = 2, durationInMinutes = random.nextInt(13), SUCCESSFUL)
    createCurrentTask(apps("PetPortal"), "1.0", envs("Ops/Prod/PROD"), days = 1, durationInMinutes = random.nextInt(6), SUCCESSFUL)
    createCurrentTask(apps("StockTrader"), "5.0", envs("Dev/DEV-DOTNET"), days = 1, durationInMinutes = random.nextInt(11), SUCCESSFUL)
    createCurrentTask(apps("NerdDinner"), "2.0", envs("Ops/Acc/ACC"), days = 1, durationInMinutes = random.nextInt(11), FAILED)
    createCurrentTask(apps("PetClinic-ear"), "2.0", envs("Dev/TEST"), days = 1, durationInMinutes = random.nextInt(4), ABORTED)
  }

  private def randomState(s1: TaskState, s2: TaskState): TaskState = {
    if (random.nextBoolean()) s1 else s2
  }

  private def generateSixMonthData(): Unit = {
    val appIds = List(
      "PetPortal",
      "StockTrader",
      "NerdDinner",
      "PetClinic-ear",
      "PetClinic-war"
    ).map("Applications/" + _)
    val envIds = List(
      "Ops/Acc/ACC",
      "Dev/DEV-DOTNET",
      "Dev/DEV",
      "Dev/TEST"
    ).map("Environments/" + _)
    val apps = readCisAsMap(appIds, repository)
    val envs = readCisAsMap(envIds, repository)
    // This data is meant to show a progression of more and more successful deployments over a 6 month period
    for (i <- 1 to 6) {
      for (_ <- 0 to (35 - (i * 4))) {
        createTask(apps("PetClinic-ear"), "2.0", envs("Dev/TEST"), month = i, durationInMinutes = random.nextInt(15), randomState(SUCCESSFUL, FAILED))
        createTask(apps("PetPortal"), "1.0", envs("Ops/Acc/ACC"), month = i, durationInMinutes = random.nextInt(15), randomState(SUCCESSFUL, ABORTED))
        createTask(apps("PetClinic-war"), "1.0", envs("Dev/DEV"), month = i, durationInMinutes = random.nextInt(8), randomState(SUCCESSFUL, FAILED))
        createTask(apps("NerdDinner"), "2.0", envs("Dev/DEV-DOTNET"), month = i, durationInMinutes = random.nextInt(11), SUCCESSFUL)
        createTask(apps("StockTrader"), "5.0", envs("Dev/DEV-DOTNET"), month = i, durationInMinutes = random.nextInt(5), SUCCESSFUL)
      }
    }
  }

  def createTask(app: BaseConfigurationItem, version: String, env: BaseConfigurationItem, month: Int, durationInMinutes: Int, state: TaskState): Unit = {
    val startDate = new DateTime().minusMonths(month)
    val completionDate = startDate.plusMinutes(durationInMinutes)

    task { t =>
      (state match {
        case SUCCESSFUL => t.success
        case FAILED => t.successWithFails
        case ABORTED => t.failed
      }).forApp(app.getName, app.get$internalId)
        .forVersion(version)
        .forEnv(env.getName, env.get$internalId)
        .startedAt(startDate)
        .completedAt(completionDate)
        .withId(UUID.randomUUID.toString)
        .withDefaultBlock
    }
  }

  def createCurrentTask(app: BaseConfigurationItem, version: String, env: BaseConfigurationItem, days: Int, durationInMinutes: Int, state: TaskState): Unit = {
    val startDate = new DateTime().minusDays(days)
    val completionDate = startDate.plusMinutes(durationInMinutes)

    task { t =>
      (state match {
        case SUCCESSFUL => t.success
        case FAILED => t.successWithFails
        case ABORTED => t.failed
      }).forApp(app.getName, app.get$internalId)
        .forVersion(version)
        .forEnv(env.getName, env.get$internalId)
        .startedAt(startDate)
        .completedAt(completionDate)
        .withId(UUID.randomUUID.toString)
        .withDefaultBlock
    }
  }

}
