package com.xebialabs.deployit.test.support.onthefly;

import static com.xebialabs.deployit.test.support.utils.VelocityUtils.evaluateTemplate;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.apache.velocity.VelocityContext;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;

/**
 * Constructs an EAR for testing, accepting multiple {@link WarOnTheFly WARs} as entries. 
 * The context root for a WAR in the EAR is its name. The actual EAR file is finally written 
 * to a temporary directory. Example:
 * 
 * <pre>
 * EarOnTheFly earOnTheFly = new EarOnTheFly("myEar");
 * earOnTheFly.addFile("foo", someFile);
 * earOnTheFly.addFile("bar", someResource);
 * ...
 * WarOnTheFly warOnTheFly = new WarOnTheFly("myWar");
 * warOnTheFly.addServlet(FooServlet.class);
 * ...
 * earOnTheFly.addWarOnTheFly(warOnTheFly); // will have context root http://myhost:myport/myWar
 * ...
 * File ear = earOnTheFly.writeToTemporaryFile(); // something like myEarXX.ear
 * </pre>
 * 
 * See also &quot;<a href="http://blog.xebia.com/2009/12/14/middleware-integration-testing-with-junit-maven-and-vmware-part-2/">
 * Middleware integration testing with JUnit, Maven and VMware: part 2</a>&quot;.
 * 
 * @see WarOnTheFly
 * @see JarOnTheFly
 */
public class EarOnTheFly extends JarOnTheFly {

	private String name;

	private Map<String, String> wars;

	public EarOnTheFly(String name) {
		this.name = name;
		this.wars = new HashMap<String, String>();
	}

	public void addWarOnTheFly(WarOnTheFly wotf) throws IOException {
		String warFilename = wotf.getName() + ".war";
		Resource warFile = new FileSystemResource(wotf.writeToTemporaryFile());
		addFile(warFilename, warFile);
		wars.put(wotf.getName(), warFilename);
	}

	public File writeToTemporaryFile() throws IOException {
		addApplicationXml();
		return writeToTemporaryFile(name, ".ear");
	}

	private void addApplicationXml() {
		VelocityContext context = new VelocityContext();
		context.put("name", name);
		context.put("wars", wars);
		String webxml = evaluateTemplate(context, "com/xebialabs/deployit/test/support/onthefly/application.xml.vm");
		addFile("META-INF/application.xml", new ByteArrayResource(webxml.getBytes()));
	}

	public String getName() {
		return name;
	}

}
