package com.xebialabs.deployit.test.support.onthefly;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;

import org.apache.commons.io.IOUtils;
import org.springframework.core.io.Resource;

/**
 * Constructs a JAR for testing, accepting multiple {@link Resource resources} as entries
 * and writing the actual JAR file to a temporary directory. Example:
 * 
 * <pre>
 * JarOnTheFly jarOnTheFly = new JarOnTheFly();
 * jarOnTheFly.addFile("foo", someFile);
 * jarOnTheFly.addFile("bar", someResource);
 * ...
 * File jar = jarOnTheFly.writeToTemporaryFile("myJar", ".jar"); // something like myJarXX.jar
 * </pre>
 * 
 * See also &quot;<a href="http://blog.xebia.com/2009/12/14/middleware-integration-testing-with-junit-maven-and-vmware-part-2/">
 * Middleware integration testing with JUnit, Maven and VMware: part 2</a>&quot;.
 */
public class JarOnTheFly {

	private Map<String, Resource> files = new HashMap<String, Resource>();

	public void addFile(String filename, Resource resource) {
		files.put(filename, resource);
	}

	public void write(File jarFile) throws IOException {
		FileOutputStream jarFileOut = new FileOutputStream(jarFile);
		try {
			JarOutputStream jarOut = new JarOutputStream(jarFileOut);
			try {
				for (Map.Entry<String, Resource> eachFile : files.entrySet()) {
					String filename = eachFile.getKey();
					Resource resource = eachFile.getValue();
					jarOut.putNextEntry(new JarEntry(filename));
					InputStream resourceIn = resource.getInputStream();
					try {
						IOUtils.copy(resourceIn, jarOut);
					} finally {
						IOUtils.closeQuietly(resourceIn);
					}
					jarOut.closeEntry();
				}
			} finally {
				IOUtils.closeQuietly(jarOut);
			}
		} finally {
			IOUtils.closeQuietly(jarFileOut);
		}
	}

	protected File writeToTemporaryFile(String prefix, String suffix) throws IOException {
		File tempJarFile = File.createTempFile(prefix, suffix);
		write(tempJarFile);
		return tempJarFile;
	}
}
