package com.xebialabs.deployit.test.support.utils;

import static org.junit.Assert.assertEquals;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;
import org.apache.commons.collections.Transformer;
import org.junit.ComparisonFailure;

import com.xebialabs.deployit.ChangeResolution;
import com.xebialabs.deployit.RunBook;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.util.ExtendedStringUtils;

/**
 * Provides assertions frequently used when unit testing {@link RunBook runbooks}.
 */
public class RunBookTestUtils {

	public static List<Step> assertOneResolutionAndGetItsSteps(Collection<ChangeResolution> resolutions) {
		return assertResolutionLengthAndGetItsSteps(resolutions, 1);
	}

	public static List<Step> assertResolutionLengthAndGetItsSteps(Collection<ChangeResolution> resolutions, int numberOfResolutions) {
		assertEquals(numberOfResolutions, resolutions.size());
		List<Step> allSteps = new ArrayList<Step>();
		for (ChangeResolution resolution : resolutions) {
			allSteps.addAll(resolution.getSteps());
		}
		return allSteps;
	}

	public static <T> T assertStepOccursOnce(List<Step> steps, final Class<T> stepClass) {
		Collection<T> stepsOfClass = getStepsOfClass(steps, stepClass);
		assertEquals(1, stepsOfClass.size());
		return stepsOfClass.iterator().next();
	}

	@SuppressWarnings("unchecked")
	public static <T> Collection<T> getStepsOfClass(List<Step> steps, Class<T> stepClass) {
		return (Collection<T>) CollectionUtils.select(steps, getStepClassPredicate(stepClass));
	}

	public static <T> Predicate getStepClassPredicate(final Class<T> stepClass) {
		return new Predicate() {
			public boolean evaluate(Object input) {
				return input.getClass() == stepClass;
			}
		};
	}

	public static <T> void assertTypeSequence(List<T> items, final Class<?>... classes) {
		if (!containsTypeSequence(items, classes)) {
			String message = "Actual type sequence does not match expected type sequence";
			String expectedClasses = classesToString(classes);
			String actualClasses = itemsClassNamesToString(items);
			throw new ComparisonFailure(message, expectedClasses, actualClasses);
		}
	}

	// assumes that the *only* items left over in items after filtering on type
	// are the ones looked for
	@SuppressWarnings("unchecked")
	private static <T> boolean containsTypeSequence(List<T> items, final Class<?>... types) {
		List<T> filteredItems = (List<T>) CollectionUtils.select(items, new Predicate() {
			public boolean evaluate(Object object) {

				if (object == null) {
					return false;
				}

				Class<?> objectClass = object.getClass();
				for (Class<?> requiredType : types) {
					if (requiredType.isAssignableFrom(objectClass)) {
						return true;
					}
				}
				return false;
			}
		});

		if (filteredItems.size() != types.length) {
			return false;
		}

		for (int i = 0; i < filteredItems.size(); i++) {
			if (!types[i].isAssignableFrom(filteredItems.get(i).getClass())) {
				return false;
			}
		}
		return true;
	}

	private static String classesToString(Class<?>[] classes) {
		return ExtendedStringUtils.join(Arrays.asList(classes), new Transformer() {
			public Object transform(Object clazz) {
				return ((Class<?>) clazz).getSimpleName();
			}
		});
	}

	private static String itemsClassNamesToString(List<?> steps) {
		return ExtendedStringUtils.join(steps, new Transformer() {
			public Object transform(Object step) {
				return step.getClass().getSimpleName();
			}
		});
	}
}
