/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.test.support.utils;

import com.meterware.httpunit.GetMethodWebRequest;
import com.meterware.httpunit.WebConversation;
import com.meterware.httpunit.WebRequest;
import com.meterware.httpunit.WebResponse;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.ci.artifact.Ear;
import com.xebialabs.deployit.ci.artifact.War;
import com.xebialabs.deployit.test.support.onthefly.EarOnTheFly;
import com.xebialabs.deployit.test.support.onthefly.WarOnTheFly;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.core.io.Resource;
import org.xml.sax.SAXException;

import javax.servlet.Servlet;
import java.io.*;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

public class ItestUtils {
	public static final String ITEST_EAR_NAME = "itest-ear";
	public static final String ITEST_WAR_NAME = "itest";
	public static final String ITEST_SUCCESS_MESSAGE = "The itest has succeeded!";

	public static File resourceToFile(Resource resource) throws IOException {
		File resourceTempFile = File.createTempFile("junit-resource", ".dat");
		resourceTempFile.deleteOnExit();
		OutputStream tempFileOut = new FileOutputStream(resourceTempFile);
		try {
			InputStream resourceIn = resource.getInputStream();
			try {
				IOUtils.copy(resourceIn, tempFileOut);
				return resourceTempFile;
			} finally {
				resourceIn.close();
			}
		} finally {
			tempFileOut.close();
		}
	}

	/**
	 * Creates an actual EAR file in a temporary location that contains a servlet of the given class, mapped to the
	 * context root given by {@link #ITEST_WAR_NAME}.
	 * <p>
	 * Returns an {@link Ear} CI whose backing file is the EAR file created.
	 * 
	 * @param itestServletClass
	 *            the class of the servlet to be deployed
	 * @return an {@code Ear} CI representing the created EAR file
	 * @throws IOException
	 *             if the EAR file could not be written
	 */
	public static Ear createItestEar(Class<? extends Servlet> itestServletClass) throws IOException {
		WarOnTheFly wotf = new WarOnTheFly(ITEST_WAR_NAME);
		wotf.addServlet(itestServletClass);
		EarOnTheFly eotf = new EarOnTheFly(ITEST_EAR_NAME);
		eotf.addWarOnTheFly(wotf);

		Ear itestEar = new Ear();
		itestEar.setLabel(eotf.getName());
		itestEar.setName(eotf.getName());
		itestEar.setLocation(eotf.writeToTemporaryFile().getPath());
		return itestEar;
	}

	/**
	 * Creates an actual WAR file in a temporary location that contains a servlet of the given class, mapped to the
	 * context root given by {@link #ITEST_WAR_NAME}.
	 * <p>
	 * Returns an {@link War} CI whose backing file is the EAR file created.
	 * 
	 * @param itestServletClass
	 *            the class of the servlet to be deployed
	 * @return an {@code War} CI representing the created EAR file
	 * @throws IOException
	 *             if the WAR file could not be written
	 */
	public static War createItestWar(Class<? extends Servlet> itestServletClass) throws IOException {
		WarOnTheFly wotf = new WarOnTheFly(ITEST_WAR_NAME);
		wotf.addServlet(itestServletClass);

		War warCi = new War();
		warCi.setLabel(wotf.getName());
		warCi.setName(wotf.getName());
		warCi.setLocation(wotf.writeToTemporaryFile().getPath());
		return warCi;
	}

	public static void assertItestServletSucceeds(Host serverHost, int port, Class<? extends Servlet> itestServletClass) throws IOException, SAXException {
		assertItestServletSucceeds(serverHost, port, itestServletClass, null);
	}

	public static void assertItestServletSucceeds(Host serverHost, int port, Class<? extends Servlet> itestServletClass, String action) throws IOException,
			SAXException {
		String url = "http://" + serverHost.getAddress() + ":" + port + "/" + ITEST_WAR_NAME + "/" + itestServletClass.getSimpleName();

		if (action != null) {
			url = url + "?action=" + action;
		}

		WebConversation wc = new WebConversation();
		WebRequest req = new GetMethodWebRequest(url);
		WebResponse resp = wc.getResponse(req);
		String responseText = resp.getText();
		assertTrue(responseText.contains(ITEST_SUCCESS_MESSAGE));
	}

	public static void assertStepSucceeds(Step step) {
		assertStepReturns(step, true);
	}

	public static void assertStepFails(Step step) {
		assertStepReturns(step, false);
	}

	private static StepExecutionContext context;

	public static StepExecutionContext getDebugStepExecutionContext() {
		if (context == null) {
			context = new DebugStepExecutionContext();
		}
		return context;
	}

	public static void clearStepExecutionContext() {
		if (context != null) {
			((DebugStepExecutionContext) context).destroy();
			context = null;
		}
	}

	public static void assertStepReturns(Step step, boolean expectedReturnValue) {
		boolean actualReturnValue = step.execute(getDebugStepExecutionContext());
		assertEquals("Step " + step.getDescription() + " did not complete with the expected state", expectedReturnValue, actualReturnValue);
	}

	public static boolean isItestEnabled(String itestName, boolean defaultValue) {
		String enabledItestsSpecification = System.getProperty("itests");
		if (StringUtils.isBlank(enabledItestsSpecification)) {
			return defaultValue;
		}

		String[] enabledItests = StringUtils.split(enabledItestsSpecification);
		return ArrayUtils.contains(enabledItests, itestName);
	}

}
