/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.reflect;

import java.io.Serializable;

import com.xebialabs.xlplatform.documentation.PublicApiRef;

/**
 * Encapsulates the metadata about the type of a {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem},
 * such as: prefix, name, type hierarchy, the source registry etc.
 *
 * This class also provides static methods to retrieve the Type of a CI by name or by class from the DescriptorRegistry.
 */
@SuppressWarnings("serial")
@PublicApiRef
public final class Type implements Serializable {

    private String prefix;
    private String name;

    private DescriptorRegistryId typeSource;

    Type(String prefix, String name, DescriptorRegistryId id) {
        if (prefix == null || name == null) {
            throw new NullPointerException("Either prefix or name is null, type should be fully qualified.");
        }
        this.prefix = prefix;
        this.name = name;
        this.typeSource = id;
    }

    /**
     *
     * @param ciClass A {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem} class object.
     * @return The Type representation of the given CI class.
     */
    public static Type valueOf(Class<?> ciClass) {
        return DescriptorRegistry.getDefaultDescriptorRegistry().lookupType(ciClass);
    }

    /**
     * @param typeName The prefixed name of a  {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem} class.
     * @return The Type representation of the given CI class prefixed name.
     */
    public static Type valueOf(String typeName) {
        return DescriptorRegistry.getDefaultDescriptorRegistry().lookupType(typeName);
    }

    /**
     * @param prefix The CI namespace for the specified {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem}.
     * @param simpleName The simple name for the specified {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem}.
     * @return The Type representation of the given CI.
     */
    public static Type valueOf(String prefix, String simpleName) {
        return DescriptorRegistry.getDefaultDescriptorRegistry().lookupType(prefix, simpleName);
    }

    /**
     * @return The metadata about this {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem}.
     */
    public Descriptor getDescriptor() {
        return DescriptorRegistry.getDescriptor(this);
    }

    /**
     *
     * @param superType The super type against which to verify the inheritance relationship.
     * @return Whether this type is a subtype of the given type.
     */
    public boolean isSubTypeOf(Type superType) {
        return DescriptorRegistry.getSubtypes(superType).contains(this);
    }

    /**
     * @param selfOrSuperType The type against which to verify the "instance of" relationship
     * @return Whether this type is a instance of the given type.
     */
    public boolean instanceOf(Type selfOrSuperType) {
        return equals(selfOrSuperType) || DescriptorRegistry.getSubtypes(selfOrSuperType).contains(this);
    }

    /**
     * @param subType The sub type against which to verify the inheritance relationship.
     * @return Whether this type is a super type of the given type.
     */
    public boolean isSuperTypeOf(Type subType) {
        return DescriptorRegistry.getSubtypes(this).contains(subType);
    }

    /**
     * @return Whether this type is a registered CI.
     */
    public boolean exists() {
        return DescriptorRegistry.exists(this);
    }

    /**
     * @return The CI namespace for this type.
     */
    public String getPrefix() {
        return prefix;
    }


    /**
     * @return The simple name for this type.
     */
    public String getName() {
        return name;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Type type1 = (Type) o;

        return prefix.equals(type1.prefix) && name.equals(type1.name);
    }

    @Override
    public int hashCode() {
        int result = prefix != null ? prefix.hashCode() : 0;
        result = 31 * result + (name != null ? name.hashCode() : 0);
        return result;
    }

    @Override
    public String toString() {
        return prefix + "." + name;
    }

    /**
     *
     * @return The id of the descriptor registry.
     */
    public DescriptorRegistryId getTypeSource() {
        return typeSource;
    }
}
