/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.validation;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;

/**
 * The annotated element must contain the specified values.
 * <p>
 * Supported types are:
 * <ul>
 *      <li>{@code STRING}</li>
 *      <li>{@code SET_OF_STRING}</li>
 *      <li>{@code LIST_OF_STRING}</li>
 *      <li>{@code MAP_STRING_STRING}</li>
 * </ul>
 * </p>
 * {@code null} elements are considered valid.
 */
@Retention(RetentionPolicy.RUNTIME)
@Rule(clazz = Contain.Validator.class, type = "contain")
@ApplicableTo({PropertyKind.STRING, PropertyKind.SET_OF_STRING, PropertyKind.LIST_OF_STRING, PropertyKind.MAP_STRING_STRING})
@Target(ElementType.FIELD)
public @interface Contain {
    String DEFAULT_MESSAGE = "The property must contain %s.";

    String message() default DEFAULT_MESSAGE;

    String[] value() default { };

    class Validator implements com.xebialabs.deployit.plugin.api.validation.Validator<Object> {
        private String message = Contain.DEFAULT_MESSAGE;

        private String[] value = { };

        @Override
        public void validate(final Object object, final ValidationContext context) {
            List<String> values = Arrays.asList(value);
            String logFriendlyValues = String.join(", ", values).replaceFirst(",(?!.*,)", " and");
            if (object != null) {
                if (object instanceof String) {
                    String s = (String) object;
                    if (!values.stream().allMatch(s::contains)) {
                        context.error(message, logFriendlyValues);
                    }
                } else if (object instanceof Collection) {
                    if (!((Collection<?>) object).containsAll(values)) {
                        context.error(message, logFriendlyValues);
                    }
                } else if (object instanceof Map) {
                    if (!((Map<?, ?>) object).keySet().containsAll(values)) {
                        context.error(message, logFriendlyValues);
                    }
                }
            }
        }

        public String getMessage() {
            return message;
        }

        public String[] getValue() {
            return value;
        }
    }
}
