/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.reflect;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.xebialabs.deployit.plugin.api.udm.Prefix;
import com.xebialabs.xlplatform.documentation.PublicApiRef;

import static java.lang.String.format;

/**
 * A registry for the {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem}s the
 * XL Deploy system should know about. Every CI registry must extend this class, and it can be added to the collection
 * of known registries, which is kept in this class.
 * <p>
 * One special descriptor registry is the LocalDescriptorRegistry, which is usually the default one. Another descriptor
 * registry is the RemoteDescriptorRegistry.
 */
@PublicApiRef
public abstract class BaseDescriptorRegistry implements IDescriptorRegistry {

    private final Map<String, Type> NAME_TYPE_CACHE = new HashMap<>();

    private final Map<Class<?>, Type> CLASS_TYPE_CACHE = new HashMap<>();

    private final DescriptorRegistryId id;
    private List<DescriptorRegistryHook> hooks = new ArrayList<>();

    protected BaseDescriptorRegistry(DescriptorRegistryId id) {
        if (id == null) {
            throw new NullPointerException("The id of a DescriptorRegistry cannot be null.");
        }
        this.id = id;
    }

    /**
     * Searches in the registry for a CI type by CI class.
     *
     * @param ciClass A class instance of a CI.
     * @return The type for the given CI class.
     */
    public Type lookupType(Class<?> ciClass) {
        if (ciClass == null) {
            throw new NullPointerException("Type name may not be null");
        }

        if (CLASS_TYPE_CACHE.containsKey(ciClass)) {
            return CLASS_TYPE_CACHE.get(ciClass);
        }

        Package ciPackage = ciClass.getPackage();
        Prefix prefix = ciPackage.getAnnotation(Prefix.class);
        if (prefix == null) {
            throw new NullPointerException(format("Package [%s] should have an @Prefix annotation for ci-class [%s]", ciPackage.getName(), ciClass.getName()));
        }
        String simpleName = ciClass.getSimpleName();
        if (simpleName.isEmpty()) {
            throw new IllegalArgumentException(format("Could not get a typename for ci-class [%s]", ciClass.getName()));
        }
        Type type = lookupType(prefix.value(), simpleName);
        CLASS_TYPE_CACHE.put(ciClass, type);
        return type;
    }

    /**
     * Searches in the registry for a CI type by CI prefixed type name.
     *
     * @param typeName The prefixed type name of a CI.
     * @return The type of the CI.
     */
    public Type lookupType(String typeName) {
        if (typeName == null) {
            throw new NullPointerException("Type name may not be null");
        }

        if (NAME_TYPE_CACHE.containsKey(typeName)) {
            return NAME_TYPE_CACHE.get(typeName);
        }

        if (typeName.indexOf('.') == -1) {
            throw new IllegalArgumentException(format("Type %s does not contain a prefix", typeName));
        }
        int indexOfLastDot = typeName.lastIndexOf('.');
        Type t = lookupType(typeName.substring(0, indexOfLastDot), typeName.substring(indexOfLastDot + 1));
        NAME_TYPE_CACHE.put(typeName, t);
        return t;
    }

    /**
     * Searches in the registry for a CI type by CI prefix and simple name.
     *
     * @param prefix     The prefix of a CI.
     * @param simpleName The simple name of a CI.
     * @return The type of the CI.
     */
    public Type lookupType(String prefix, String simpleName) {
        return new Type(prefix, simpleName, id);
    }

    public DescriptorRegistryId getId() {
        return id;
    }

    @Override
    public void verifyTypes() {
        // override if needed
    }

    @Override
    public void registerHook(final DescriptorRegistryHook hook) {
        this.hooks.add(hook);
    }

    @Override
    public List<DescriptorRegistryHook> getHooks() {
        return hooks;
    }
}
