/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.udm;

import com.xebialabs.xlplatform.documentation.PublicApiRef;

import java.util.*;

import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.ENVIRONMENTS;

/**
 * A Dictionary is a CI that contains {@link Environment}-specific entries for placeholder resolution.
 *
 * A dictionary value can refer to another dictionary entry. This is accomplished by using the {{..}} placeholder syntax.
 * For example:
 * <code>
 *     MESSAGE	Welcome to {{APP_NAME}}!
 * </code>
 */
@SuppressWarnings("serial")
@Metadata(root = ENVIRONMENTS, description = "A Dictionary contains key-value pairs that can be replaced")
@TypeIcon(value = "icons/types/udm.Dictionary.svg")
@PublicApiRef
public class Dictionary extends AbstractDictionary implements IEncryptedDictionary {

    @Property(description = "The dictionary entries", required = false)
    private Map<String, String> entries = new HashMap<>();

    @Property(description = "The encrypted dictionary entries", required = false, password = true)
    private Map<String, String> encryptedEntries = new HashMap<>();

    /**
     * @param dict The dictionary entries.
     */
    public void setEntries(Map<String, String> dict) {
        this.entries = dict;
    }

    /**
     * @return The dictionary entries.
     */
    @Override
    public Map<String, String> getEntries() {
        return entries;
    }

    /**
     * @return The encrypted dictionary entries
     */
    @Override
    public Map<String, String> getEncryptedEntries() {
        return encryptedEntries;
    }

    /**
     * @param encryptedEntries The encrypted dictionary entries
     */
    public void setEncryptedEntries(Map<String, String> encryptedEntries) {
        this.encryptedEntries = encryptedEntries;
    }

    /**
     * @param key The key of a dictionary entry.
     * @return The value associated with the given key.
     */
    @Override
    public String getValue(String key) {
        return entries.get(key);
    }

    /**
     * @param key The key of a dictionary entry.
     * @return Whether the dictionary contains the given key.
     */
    public boolean hasValue(String key) {
        return entries.containsKey(key);
    }

    /**
     * @return The set of {@link Application}s only for which the dictionary is applied.
     */
    public Set<Application> getRestrictToDistributions() {
        return getRestrictToApplications();
    }
}
