/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.reflect;

import java.util.Map;

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Parameters;
import com.xebialabs.xlplatform.documentation.PublicApiRef;

/**
 * Encapsulates the metadata about a control task (also named control method) of a type in the XL Deploy type system.
 *
 * Control tasks allow actions to be executed on CIs and can be invoked from the GUI or the CLI. Control tasks specify
 * a list of steps to be executed in order. Control tasks can be parametrized in two ways:
 * <ol>
 *  <li>by specifying arguments to the control task in the control task configuration</li>
 *  <li>by allowing the user to specify parameters to the control task during control task execution</li>
 * </ol>
 *
 * Arguments are configured in the control task definition in the synthetic.xml file. Arguments are specified as
 * attributes on the synthetic method definition XML and are passed as-is to the control task.
 *
 * Parameters are specified by defining a parameters CI type, which needs to extend the {@link com.xebialabs.deployit.plugin.api.udm.Parameters} CI.
 */
@PublicApiRef
public interface MethodDescriptor {

    /**
     * @return The name of the control task.
     */
    String getName();

    /**
     * @return Label text for use in the UI.
     */
    String getLabel();

    /**
     * @return Textual description of the control task.
     */
    String getDescription();

    /**
     * @return A map of arguments to be passed to the method call. An entry key represents the name of the argument,
     * and an entry value is the value of the argument.
     */
    Map<String, String> getAttributes();

    /**
     * @return The metadata {@link Type} object about the specified {@link com.xebialabs.deployit.plugin.api.udm.Parameters} CI.
     */
    Type getParameterObjectType();

    /**
     * @param item The {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem} on which to invoke the control task.
     * @param params The specified {@link com.xebialabs.deployit.plugin.api.udm.Parameters}s.
     * @param <T> The type of the returned object.
     * @return The result of execution of type T.
     */
    <T> T invoke(ConfigurationItem item, Parameters params);

    /**
     * @param item The {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem} on which to invoke the control task.
     * @param <T> The type of the returned object.
     * @return The result of execution of type T.
     */
    <T> T invoke(ConfigurationItem item);

    /**
     * @return The fully qualified name of the control task. This is the type name followed by a dot and then
     * the control task name. For example: overthere.SshJumpstation.checkConnection
     */
    String getFqn();
}
