package com.xebialabs.deployit.plugin.api.udm.artifact;

import com.google.common.base.Function;
import com.google.common.collect.MapMaker;
import com.google.common.collect.Sets;
import com.google.common.io.Closeables;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.overthere.local.LocalFile;
import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.file.TFileReader;
import de.schlichtherle.truezip.file.TFileWriter;
import de.schlichtherle.truezip.fs.FsSyncException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.lang.Math.abs;
import static java.util.regex.Pattern.CASE_INSENSITIVE;
import static java.util.regex.Pattern.COMMENTS;

public class Artifacts {

	public static Map<String, Pattern> patternMap = new MapMaker().makeComputingMap(new Function<String, Pattern>() {
        public Pattern apply(String regex) {
	        return Pattern.compile(regex, COMMENTS | CASE_INSENSITIVE);
        }
	});

	public static void scanPlaceholders(SourceArtifact artifact, PlaceholderScanner scanner) {
		checkArgument(artifact.getFile() != null, artifact + " has no file");
		checkArgument(artifact.getFile() instanceof LocalFile, "Cannot scan for placeholders in " + artifact + " because its file is a " + artifact.getFile().getClass().getName() + " and not a " + LocalFile.class.getName());
        
		artifact.setPlaceholders(Sets.<String> newTreeSet());
		if (artifact.hasProperty("scanPlaceholders") && !(Boolean) artifact.getProperty("scanPlaceholders")) {
			return;
		}
        TFile from = toTFile(artifact.getFile());
		try {
            doScanPlaceholders(artifact, from, scanner);
		} finally {
			umountQuietly(from);
		}
	}

	private static void doScanPlaceholders(SourceArtifact artifact, TFile from, PlaceholderScanner scanner) {
		if (from.isDirectory()) {
			for (TFile f : from.listFiles()) {
				doScanPlaceholders(artifact, f, scanner);
			}
		} else if (isTextFile(from, artifact.getTextFileNamesRegex())) {
			artifact.getPlaceholders().addAll(readPlaceholders(from, scanner));
		}
    }

	private static Set<String> readPlaceholders(final TFile from, final PlaceholderScanner scanner) {
		Reader in = null;
		try {
            logger.trace("Reading placeholders from path {}, file {}", from.getPath(), from.getName());
			in = new TFileReader(from);
			return scanner.scan(in);
		} catch (IOException exc) {
			throw new RuntimeIOException("Cannot scan for placeholders in " + from, exc);
		} catch (RuntimeException exc) {
			throw new RuntimeException("Cannot scan for placeholders in " + from, exc);
		} finally {
			Closeables.closeQuietly(in);
		}
	}

	private static boolean isTextFile(final TFile f, final String textFileNamesRegex) {
		checkNotNull(textFileNamesRegex, "Regex is null");

		Pattern textFileNamesPattern = patternMap.get(textFileNamesRegex);
		Matcher textFileNamesMatcher = textFileNamesPattern.matcher(f.getName());
		boolean isTextFile = textFileNamesMatcher.matches();
		logger.debug("Determined {} to be a {} file", f.getName(), isTextFile ? "text" : "binary");

		return isTextFile;
	}

	public static void replacePlaceholders(DerivedArtifact<? extends SourceArtifact> derivedArtifact, PlaceholderReplacer replacer) {
		if(derivedArtifact.getSourceArtifact() == null) {
			derivedArtifact.setFile(null);
		} else {
			checkArgument(derivedArtifact.getSourceArtifact() instanceof SourceArtifact, derivedArtifact.getSourceArtifact() + " is not an Artifact");
			checkArgument(derivedArtifact.getSourceArtifact().getFile() != null, derivedArtifact.getSourceArtifact() + " has no file");
			checkArgument(derivedArtifact.getSourceArtifact().getFile() instanceof LocalFile, "Cannot replace placeholders in" + derivedArtifact.getSourceArtifact()
			        + " because its file is a " + derivedArtifact.getSourceArtifact().getFile().getClass().getName() + " and not a " + LocalFile.class.getName());

			TFile from = toTFile(derivedArtifact.getSourceArtifact().getFile());
			try {
				boolean isBinaryFile = from.isFile() && !isTextFile(from, getTextFileNamesRegex(derivedArtifact));

				if (derivedArtifact.getPlaceholders().isEmpty() || isBinaryFile) {
					derivedArtifact.setFile(derivedArtifact.getSourceArtifact().getFile());
				} else {
					TFile to = getOutputFile(derivedArtifact);

					doReplacePlaceholders(derivedArtifact, from, to, replacer);

					File deployedFile = saveArchive(to);

					derivedArtifact.setFile(LocalFile.valueOf(deployedFile));
				}
			} finally {
				umountQuietly(from);
			}
		}
	}

	private static void doReplacePlaceholders(DerivedArtifact<? extends SourceArtifact> derivedArtifact, TFile from, TFile to, PlaceholderReplacer replacer) {
		if (from.isDirectory()) {
	        try {
	        	to.mkdir(false);
	        } catch(IOException exc) {
	        	throw new RuntimeIOException("Cannot create directory " + to, exc);
	        }

	        for (TFile f : from.listFiles()) {
				TFile t = new TFile(to, f.getName());
				doReplacePlaceholders(derivedArtifact, f, t, replacer);
			}
		} else if (isTextFile(from, getTextFileNamesRegex(derivedArtifact))) {
			replace(from, to, replacer, derivedArtifact.getPlaceholders());
		} else {
			try {
                from.cp_p(to);
            } catch (IOException exc) {
            	throw new RuntimeIOException("Cannot copy " + from + " to " + to, exc);
            }
		}
	}

	private static void replace(final TFile from, final TFile to, final PlaceholderReplacer replacer, Map<String, String> resolution) {
		Reader reader = null;
		Writer writer = null;
		try {
			reader = new TFileReader(from);
			writer = new TFileWriter(to);
			replacer.replace(reader, writer, resolution);
		} catch (IOException exc) {
			throw new RuntimeIOException("Cannot copy " + from + " to " + to + " while replacing placeholders", exc);
		} finally {
			Closeables.closeQuietly(reader);
			Closeables.closeQuietly(writer);
		}
	}

	private static File saveArchive(TFile outputArchive) {
		if (outputArchive.isArchive() && outputArchive.getEnclArchive() == null && outputArchive.isDirectory()) {
			try {
				TFile.umount(outputArchive);
			} catch (IOException exc) {
				throw new RuntimeIOException("Cannot write archive " + outputArchive, exc);
			}
        }

		// Return a regular java.io.File pointing to the file, folder or archive just written
		return new File(outputArchive.getPath());
    }

	private static String getTextFileNamesRegex(DerivedArtifact<? extends SourceArtifact> derivedArtifact) {
		return ((SourceArtifact) derivedArtifact.getSourceArtifact()).getTextFileNamesRegex();
	}


	private static final TFile getOutputFile(DerivedArtifact<? extends SourceArtifact> derivedArtifact) {
		OverthereFile workDir = derivedArtifact.getSourceArtifact().getFile().getParentFile();

		Random r = new Random();
		String name = derivedArtifact.getName();
		for(;;) {
			OverthereFile deployedArchiveDir = workDir.getFile(name);

			if(!deployedArchiveDir.exists()) {
				deployedArchiveDir.mkdir();
				return toTFile(deployedArchiveDir.getFile(derivedArtifact.getSourceArtifact().getFile().getName()));
			}

			name = derivedArtifact.getName() + abs(r.nextInt());
		}
	}

	private static TFile toTFile(OverthereFile file) {
		return new TFile(((LocalFile) file).getFile());
	}

	private static void umountQuietly(TFile from) {
		if (from.isArchive()) {
			try {
				TFile.umount(from);
			} catch (FsSyncException e) {
				logger.error("Could not umount archive {}", from);
				logger.debug("Exception: ", e);
			}
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(Artifacts.class);

}
