package com.xebialabs.deployit.plugin.api.deployment.execution;

import static com.google.common.collect.Lists.newArrayList;
import static java.util.Collections.unmodifiableList;

import java.util.ArrayList;
import java.util.List;
import java.util.TreeSet;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimaps;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;

/**
 * The lowest type of sub-plan, this plan contains {@link DeploymentStep}s which can be executed and {@link Delta} objects
 * for which this InterleavedPlan has been built.
 *
 * The {@link DeploymentStep}s in the InterleavedPlan are sorted by the {@link StepComparator}
 */
public class InterleavedPlan implements Plan {
    private List<Delta> deltas;
    private ListMultimap<Integer, DeploymentStep> steps;

	InterleavedPlan(List<Delta> deltas) {
		this.deltas = unmodifiableList(deltas);
		this.steps = Multimaps.unmodifiableListMultimap(ArrayListMultimap.<Integer, DeploymentStep>create());
	}
	
    InterleavedPlan(Delta... deltas) {
	    this(newArrayList(deltas));
    }

	/**
	 * Constructor. Takes the steps, and sorts them in a new List, the passed in list is not structurally modified.
	 * Steps in the InterlavedPlan are sorted using the {@link StepComparator}
	 * @param deltas
	 * @param steps
	 */
    InterleavedPlan(List<Delta> deltas, ListMultimap<Integer, DeploymentStep> steps) {
        this.deltas = unmodifiableList(deltas);
        this.steps = Multimaps.unmodifiableListMultimap(steps);
    }

	/**
	 * Gets the {@link Delta}s for which this InterleavedPlan has been compiled
	 * @return the Delta's
	 */
    public List<Delta> getDeltas() {
        return deltas;
    }

	public ListMultimap<Integer, DeploymentStep> getStepsMap() {
		return steps;
	}

	@Override
    public List<DeploymentStep> getSteps() {
		TreeSet<Integer> keys = new TreeSet<Integer>(steps.keySet());
		ArrayList<DeploymentStep> objects = newArrayList();
		for (Integer key : keys) {
			objects.addAll(steps.get(key));
		}
		return objects;
    }

    @Override
    public String getType() {
        return InterleavedPlan.class.getName();
    }
}
