package com.xebialabs.deployit.plugin.api.reflect;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static com.google.common.base.Strings.emptyToNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.google.common.collect.Sets.symmetricDifference;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.BOOLEAN;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.ENUM;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.INTEGER;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.MAP_STRING_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.STRING;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.childByName;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.forEach;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getOptionalBooleanAttribute;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getOptionalStringAttribute;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getRequiredStringAttribute;
import static com.xebialabs.deployit.plugin.api.utils.Strings.isBlank;
import static java.lang.Character.isUpperCase;
import static java.lang.Character.toUpperCase;
import static java.lang.String.format;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.util.Arrays;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

import com.google.common.base.Function;
import com.google.common.base.Predicates;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.xebialabs.deployit.plugin.api.inspection.InspectionProperty;
import com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.Closure;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.validation.ReferenceCollectionTypeValidator;
import com.xebialabs.deployit.plugin.api.validation.RequiredValidator;
import com.xebialabs.deployit.plugin.api.validation.ValidationContext;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.plugin.api.validation.Validator;

public class PropertyDescriptor {

	private Descriptor declaringDescriptor;
	private String name;

    private Field field;
	private boolean asContainment;
	private String category;
	private String description;
	private String label;
	private boolean password;
	private boolean required;
    private Property.Size size;
    private PropertyKind kind;
    private List<String> enumValues;
    private Class<?> enumClass;
    private Type referencedType;
    private boolean hidden;
	private boolean inspectionProperty;
	private boolean requiredForInspection;
	private boolean isTransient;
	private Set<Validator<?>> validationRules = newHashSet();

	private PropertyDescriptor(Field field) {
		this.name = field.getName();
		this.field = field;
		field.setAccessible(true);
	}

	public PropertyDescriptor(String name) {
		this.name = name;
		this.field = null;
	}

	PropertyDescriptor(PropertyDescriptor pd, Descriptor newOwner) {
		this.declaringDescriptor = newOwner;
		this.name = pd.name;
		this.field = pd.field;
		this.asContainment = pd.asContainment;
		this.category = pd.category;
		this.description = pd.description;
		this.label = pd.label;
		this.password = pd.password;
		this.required = pd.required;
		this.size = pd.size;
		this.kind = pd.kind;
		this.enumValues = pd.enumValues;
		this.enumClass = pd.enumClass;
		this.referencedType = pd.referencedType;
		this.hidden = pd.hidden;
		this.inspectionProperty = pd.inspectionProperty;
		this.requiredForInspection = pd.requiredForInspection;
		this.validationRules = newHashSet(pd.validationRules);
		isTransient = pd.isTransient();
		// Clone the default value...
		GlobalContext.register(this, pd);
	}

	static PropertyDescriptor from(Descriptor descriptor, Field field) {
		Property annotation = field.getAnnotation(Property.class);

		PropertyDescriptor propertyDescriptor = new PropertyDescriptor(field);
		propertyDescriptor.declaringDescriptor = descriptor;
		propertyDescriptor.initMetadata(annotation);
		propertyDescriptor.initType(annotation);
		initInspectionMetadata(field, propertyDescriptor);
		initValidationMetadata(field, propertyDescriptor);
		verify(propertyDescriptor);
		return propertyDescriptor;
	}

	private static void initValidationMetadata(Field field, PropertyDescriptor propertyDescriptor) {
		propertyDescriptor.addDefaultValidationRules();
		for (Annotation annotation : field.getAnnotations()) {
			if (ValidationRuleConverter.isRule(annotation)) {
				propertyDescriptor.validationRules.add(ValidationRuleConverter.makeRule(annotation));
			}
		}
	}

	static PropertyDescriptor from(Descriptor descriptor, Element propertyElement) {
        String name = getRequiredStringAttribute(propertyElement, "name");
        
		PropertyDescriptor propertyDescriptor = new PropertyDescriptor(name);
		propertyDescriptor.declaringDescriptor = descriptor;
		propertyDescriptor.initSynthetic(propertyElement);
		verify(propertyDescriptor);
		return propertyDescriptor;
	}

	private static void verify(PropertyDescriptor propertyDescriptor) {
		verifyName(propertyDescriptor, "name");
		verifyName(propertyDescriptor, "id");
		verifyName(propertyDescriptor, "type");
		if (propertyDescriptor.hidden) {
			checkState(propertyDescriptor.isTransient, "Hidden property %s should be transient", propertyDescriptor);
		}
		if (propertyDescriptor.kind == PropertyKind.BOOLEAN && propertyDescriptor.required) {
			logger.warn("Boolean property {} should not be required.", propertyDescriptor);
			propertyDescriptor.required = false;
		}
	}

	private static void verifyName(PropertyDescriptor propertyDescriptor, String name) {
		checkArgument(!propertyDescriptor.name.equals(name), "Cannot define a property named '%s' on %s", name, propertyDescriptor.declaringDescriptor);
	}

	private static void initInspectionMetadata(Field field, PropertyDescriptor propertyDescriptor) {
		if (field.isAnnotationPresent(InspectionProperty.class)) {
			propertyDescriptor.inspectionProperty = true;
			propertyDescriptor.requiredForInspection = field.getAnnotation(InspectionProperty.class).required();
		}
	}

	private void initMetadata(Property annotation) {
		category = annotation.category();
		label = isBlank(annotation.label()) ? deCamelize(name) : annotation.label();
		description = isBlank(annotation.description()) ? label : annotation.description();
		password = annotation.password();
		required = annotation.required();
		size = annotation.size();
        hidden = annotation.hidden();
		isTransient = hidden || annotation.isTransient();
	}

	private void initType(Property annotation) {
		Class<?> type = field.getType();
		if (type == boolean.class) {
			kind = BOOLEAN;
		} else if (type == int.class) {
			kind = INTEGER;
		} else if (type == String.class) {
			kind = STRING;
		} else if (type.isEnum()) {
			kind = ENUM;
			initEnumValues(field.getType());
		} else if (ConfigurationItem.class.isAssignableFrom(type)) {
			kind = CI;
			referencedType = Type.valueOf(type);
			asContainment = annotation.asContainment();
		} else if (Set.class.isAssignableFrom(type)) {
			initSetType(annotation);
		} else if (Map.class.isAssignableFrom(type)) {
			initMapType(annotation);
		} else if (List.class.isAssignableFrom(type)){
			initListType(annotation);
		} else {
			throw new IllegalArgumentException(format("Type of %s not supported as an @Property field, found on %s.%s", type.getName(), field
			        .getDeclaringClass().getName(), name));
		}

        GlobalContext.register(this, emptyToNull(annotation.defaultValue()));
	}

	private Object convertValue(String val) {
        if (Strings.isNullOrEmpty(val)) return val;
        switch (kind) {
            case BOOLEAN:
                return Boolean.parseBoolean(val);
            case INTEGER:
                return Integer.parseInt(val);
            case STRING:
                return val;
            case ENUM:
                for (Enum<?> enumConstant : (Enum<?>[]) enumClass.getEnumConstants()) {
                    if (enumConstant.name().equalsIgnoreCase(val)) {
                        return enumConstant;
                    }
                }
                throw new IllegalArgumentException("Value " + val + " not a member of enum " + field.getType());
            case SET_OF_STRING:
			return newHashSet(splitValue(val));
            case LIST_OF_STRING:
			return newArrayList(splitValue(val));
            case MAP_STRING_STRING:
		        return decodeMap(val);
            default:
                throw new IllegalArgumentException("Property " + name + " of kind " + kind + " cannot be converted from a string value");
        }
    }

	private Iterable<String> splitValue(String val) {
	    return Splitter.on(',').trimResults().split(val);
    }

    private Map<String, String> decodeMap(String val) {
		Map<String, String> map = newHashMap();
		// TODO maybe create smart parser with quote support...
		for (String s : splitValue(val)) {
			String[] split = s.split(":");
            checkArgument(emptyToNull(split[0]) != null, "Property %s of kind %s cannot be converted from string value '%s' because of any empty key.", name, kind, val);
            if (split.length == 1) {
                map.put(split[0],"");
            } else {
			    map.put(split[0], split[1]);
            }
		}
		return map;
	}

	private void initSetType(Property annotation) {
		Class<?> setType = getGenericType(Set.class, 1, 0);
		if (setType == String.class) {
			kind = SET_OF_STRING;
		} else if (ConfigurationItem.class.isAssignableFrom(setType)) {
			kind = SET_OF_CI;
			referencedType = Type.valueOf(setType);
			asContainment = annotation.asContainment();
		} else {
			throw new IllegalStateException(format("Unsupported Set type encountered for [%s]. Only support String and ConfigurationItem", name));
		}
	}

	private void initMapType(Property annotation) {
		checkArgument(getGenericType(Map.class, 2, 0) == String.class, "Property %s.%s of type Map should be Map<String, String>", field.getDeclaringClass().getName(), name);
		checkArgument(getGenericType(Map.class, 2, 1) == String.class, "Property %s.%s of type Map should be Map<String, String>", field.getDeclaringClass().getName(), name);
		kind = MAP_STRING_STRING;
	}
	
	private void initListType(Property annotation) {
		Class<?> listType = getGenericType(List.class, 1, 0);
		if (listType == String.class) {
			kind = LIST_OF_STRING;
		} else if (ConfigurationItem.class.isAssignableFrom(listType)) {
			kind = LIST_OF_CI;
			referencedType = Type.valueOf(listType);
			asContainment = annotation.asContainment();
		} else {
			throw new IllegalStateException(format("Unsupported List type encountered for [%s]. Only support String and ConfigurationItem", name));
		}
	}

	private Class<?> getGenericType(Class<?> collectionClass, int nrExpectedTypes, int indexOfType) {
		java.lang.reflect.Type genericType = field.getGenericType();
		checkArgument(genericType instanceof ParameterizedType, "The field %s.%s is a %s but it isn't a generic type (%s)",
                field.getDeclaringClass().getName(), name, collectionClass, genericType);
		java.lang.reflect.Type[] actualTypeArguments = ((ParameterizedType) genericType).getActualTypeArguments();
		checkArgument(actualTypeArguments.length == nrExpectedTypes, "The field %s is a %s.%s but it doesn't have the right generic type (%s)", field.getDeclaringClass()
                .getName(), name, collectionClass, actualTypeArguments);
		checkArgument(actualTypeArguments[indexOfType] instanceof Class, "The field %s.%s is a %s but it is not a concrete subclass (%s)", field.getDeclaringClass()
                .getName(), name, collectionClass, Arrays.toString(actualTypeArguments));
		return (Class<?>) actualTypeArguments[indexOfType];

	}

	private void initEnumValues(Class<?> enumClass) {
		enumValues = newArrayList();
        this.enumClass = enumClass;
		for (Enum<?> enumValue : (Enum<?>[]) enumClass.getEnumConstants()) {
			enumValues.add(enumValue.name());
		}
	}

    void overrideWith(PropertyDescriptor superDescriptor) {
		checkArgument(
				superDescriptor.getKind() == this.getKind(),
				"Type '%s' attempts to overrides property '%s' declared in type '%s', but kind attribute does not match. Derived kind: '%s'. Super kind: '%s'.",
				getDeclaringDescriptor().getType(), getName(), 
				superDescriptor.getDeclaringDescriptor().getType(), 
				getKind(), superDescriptor.getKind());
		checkOverrideArgument(superDescriptor.isAsContainment() == this.isAsContainment(), "asContainment", superDescriptor.isAsContainment(), superDescriptor);
	    checkOverrideArgument((superDescriptor.getReferencedType() == null && this.getReferencedType() == null) ||
			    (superDescriptor.getReferencedType() != null && superDescriptor.getReferencedType().equals(this.getReferencedType())), "referenceType", superDescriptor.getReferencedType(), superDescriptor);
		checkOverrideArgument(superDescriptor.enumClass == this.enumClass, "enumClass", superDescriptor.enumClass, superDescriptor);
		checkOverrideArgument(superDescriptor.isPassword() == this.isPassword(), "password", superDescriptor.isPassword(), superDescriptor);
		if (superDescriptor.isRequired()) {
			checkOverrideArgument(this.isRequired(), "required", superDescriptor.isRequired(), superDescriptor);
		}

		field = superDescriptor.field;
    }

    private void checkOverrideArgument(boolean condition, String attribute, Object expectedValue, PropertyDescriptor superDescriptor) {
        final String attributeErrorTemplate = "Type '%s' attempts to overrides property '%s' declared in type '%s', but '%s' attribute does not match that in the super type. Should be set to %s.";
        checkArgument(condition, attributeErrorTemplate, getDeclaringDescriptor().getType(), getName(), superDescriptor.getDeclaringDescriptor().getType(), attribute, expectedValue);
    }

	private void initSynthetic(Element propertyElement) {
		kind = PropertyKind.valueOf(getOptionalStringAttribute(propertyElement, "kind", STRING.name()).toUpperCase());
		category = getOptionalStringAttribute(propertyElement, "category", "Common");
		label = getOptionalStringAttribute(propertyElement, "label", deCamelize(name));
		description = getOptionalStringAttribute(propertyElement, "description", label);
		required = getOptionalBooleanAttribute(propertyElement, "required", true);
		password = getOptionalBooleanAttribute(propertyElement, "password", false);
		asContainment = getOptionalBooleanAttribute(propertyElement, "as-containment", false);
		size = Property.Size.valueOf(getOptionalStringAttribute(propertyElement, "size", Property.Size.MEDIUM.name()).toUpperCase());
        String defaultValueAttr = getOptionalStringAttribute(propertyElement, "default", null);
        hidden = getOptionalBooleanAttribute(propertyElement, "hidden", false);
		if (hidden) {
			isTransient = true;
		} else {
            isTransient = getOptionalBooleanAttribute(propertyElement, "transient", false);
		}

		if (kind == ENUM) {
			String enumClassAttributeValue = getRequiredStringAttribute(propertyElement, "enum-class", "for property " + name + " of kind " + kind);
			try {
				enumClass = Class.forName(enumClassAttributeValue);
				if (!enumClass.isEnum()) {
					throw new IllegalArgumentException("enum-class supplied for property " + name + " of kind " + kind + " is not an enum: "
					        + enumClassAttributeValue);
				}
				initEnumValues(enumClass);
			} catch (ClassNotFoundException exc) {
				throw new IllegalArgumentException("Unknown enum-class supplied for property " + name + " of kind " + kind + ": " + enumClassAttributeValue,
				        exc);
			}
		}

		if (kind == CI || kind == SET_OF_CI || kind == LIST_OF_CI) {
            referencedType = Type.valueOf(getRequiredStringAttribute(propertyElement, "referenced-type", "for property " + name + " of kind " + kind));
		}

		initSyntheticValidationRules(propertyElement);
		addDefaultValidationRules();
        GlobalContext.register(this, defaultValueAttr);
	}

	private void addDefaultValidationRules() {
		if (required && !asContainment && kind != PropertyKind.BOOLEAN) {
			validationRules.add(new RequiredValidator());
		}
        
        if (EnumSet.of(SET_OF_CI, LIST_OF_CI).contains(kind) && !asContainment) {
            validationRules.add(new ReferenceCollectionTypeValidator(this));
        }
	}

	private void initSyntheticValidationRules(Element propertyElement) {
		forEach(childByName(propertyElement, Predicates.equalTo("rule")), new Closure<Element>() {
			public void call(Element element) {
				validationRules.add(ValidationRuleConverter.makeRule(element, getFqn()));
			}
		});
	}

	static PropertyDescriptor generateDeployableFrom(Descriptor descriptor, PropertyDescriptor deployedPropertyDescriptor) {
		PropertyDescriptor propertyDescriptor = new PropertyDescriptor(deployedPropertyDescriptor.getName());
		propertyDescriptor.declaringDescriptor = descriptor;
		propertyDescriptor.generateDeployable(deployedPropertyDescriptor);
		return propertyDescriptor;
	}

	private void generateDeployable(PropertyDescriptor deployedPropertyDescriptor) {
        // Generated 'simple' Deployable properties always are of kind String so that placeholder/dictionaries values can be filled.
		kind = deployedPropertyDescriptor.getKind().isSimple() ? PropertyKind.STRING : deployedPropertyDescriptor.getKind();
		category = deployedPropertyDescriptor.getCategory();
		label = deployedPropertyDescriptor.getLabel();
		description = format("%s (%s)", deployedPropertyDescriptor.getDescription(), deployedPropertyDescriptor.getKind().name().toLowerCase());
		required = false;
		password = deployedPropertyDescriptor.isPassword();
		size = deployedPropertyDescriptor.getSize();
        field = null;
    }

	public String getName() {
		return name;
	}

	public Field getField() {
		return field;
	}

	public String getDescription() {
		return description;
	}

	public boolean isAsContainment() {
		return asContainment;
	}

	public String getCategory() {
		return category;
	}

	public String getLabel() {
		return label;
	}

	public boolean isPassword() {
		return password;
	}

	public boolean isRequired() {
		return required;
	}

	public Property.Size getSize() {
		return size;
	}

	public PropertyKind getKind() {
		return kind;
	}

	public List<String> getEnumValues() {
		return enumValues;
	}

	public Descriptor getDeclaringDescriptor() {
		return declaringDescriptor;
	}

	public Type getReferencedType() {
		return referencedType;
	}

    public Object getDefaultValue() {
        return convertValue(GlobalContext.lookup(this));
    }

    public boolean isHidden() {
        return hidden;
    }

    @SuppressWarnings("unchecked")
    public Object get(ConfigurationItem item) {
		if (field != null) {
			try {
				return field.get(item);
			} catch (IllegalAccessException e) {
				throw new RuntimeException("Cannot get field " + field, e);
			}
		} else {
			Map<String, Object> synth = (Map<String, Object>) ReflectionUtils.getField(item, declaringDescriptor.getSyntheticPropertiesField());
			return synth.get(name);
		}
	}

	@SuppressWarnings("unchecked")
    public void set(ConfigurationItem item, Object value) {
        if (kind.isSimple() && value instanceof String) {
            value = convertValue((String) value);
        } else if (value == null) {
	        value = getDefaultValue();
        }

		try {
			if (field != null) {
				if (value == null) {
					return;
				}
				field.set(item, value);
			} else {
				Map<String, Object> synth = (Map<String, Object>) ReflectionUtils.getField(item, declaringDescriptor.getSyntheticPropertiesField());
				synth.put(name, value);
			}
		} catch (IllegalAccessException e) {
			throw new RuntimeException("Cannot set field " + field, e);
		}
	}

	public boolean areEqual(ConfigurationItem item, ConfigurationItem other) {
        return areEqual(item, other, new HashSet<String>());
	}

	boolean areEqual(ConfigurationItem item, ConfigurationItem other, Set<String> itemsBeingCompared) {
        Object left = get(item);
        Object right = get(other);
        return areValuesEqual(left, right, itemsBeingCompared);
	}

    @SuppressWarnings("unchecked")
	private boolean areValuesEqual(Object itemValue, Object otherValue, Set<String> itemsBeingCompared) {

		if (itemValue == null) {
			return otherValue == null;
		} else if (otherValue == null) {
			return false;
		}

		switch (kind) {
		case SET_OF_STRING:
			return symmetricDifference((Set<String>) itemValue, (Set<String>) otherValue).isEmpty();
		case SET_OF_CI:
            Function<ConfigurationItem, String> f = new Function<ConfigurationItem, String>() {
				@Override
				public String apply(ConfigurationItem from) {
					return from.getName();
				}
			};

            Iterable<ConfigurationItem> cis = Iterables.concat((Set<ConfigurationItem>) itemValue, (Set<ConfigurationItem>) otherValue);
            Multimap<String,ConfigurationItem> index = Multimaps.index(cis, f);
            for(String key : index.keySet()) {
                Collection<ConfigurationItem> cisToCompare = index.get(key);
                if (cisToCompare.size() != 2) {
                    return false;
                }
                Iterator<ConfigurationItem> itemIterator = cisToCompare.iterator();
                ConfigurationItem lhs = itemIterator.next();
                ConfigurationItem rhs = itemIterator.next();
                if (lhs.getType().equals(rhs.getType())) {
                    Descriptor descriptor = DescriptorRegistry.getDescriptor(lhs.getType());
                    if (descriptor.areEqual(lhs, rhs, itemsBeingCompared)) {
                        continue;
                    }
                }

                return false;
            }
            return true;
		case CI:
			ConfigurationItem itemValueAsCi = (ConfigurationItem) itemValue;
			ConfigurationItem otherValueAsCi = (ConfigurationItem) otherValue;
            if (itemValueAsCi.getName().equals(otherValueAsCi.getName())) {
                Descriptor descriptor = DescriptorRegistry.getDescriptor(itemValueAsCi.getType());
                return descriptor.areEqual(itemValueAsCi, otherValueAsCi, itemsBeingCompared);
            }
            return false;
		case MAP_STRING_STRING:
			Map<String, String> left = (Map<String, String>) itemValue;
			Map<String, String> right = (Map<String, String>) otherValue;
			return Maps.difference(left, right).areEqual();
		case LIST_OF_STRING:
			List<String> leftStrings = (List<String>) itemValue;
			List<String> rightStrings = (List<String>) otherValue;
			return leftStrings.equals(rightStrings);
		case LIST_OF_CI:
			List<ConfigurationItem> lhs = (List<ConfigurationItem>) itemValue;
			List<ConfigurationItem> rhs = (List<ConfigurationItem>) otherValue;
			if (lhs.size() != rhs.size()) {
				return false;
			} else {
				Iterator<ConfigurationItem> lIter = lhs.iterator();
				Iterator<ConfigurationItem> rIter = rhs.iterator();
				ConfigurationItem lItem;
				ConfigurationItem rItem;
				while (lIter.hasNext()) {
					lItem = lIter.next();
					rItem = rIter.next();
					if (lItem.getName().equals(rItem.getName()) && lItem.getType().equals(rItem.getType())) {
						Descriptor descriptor = DescriptorRegistry.getDescriptor(lItem.getType());
						if (descriptor.areEqual(lItem, rItem, itemsBeingCompared)) {
							continue;
						}
					}
					return false;
				}
				return true;
			}
		default:
			return itemValue.equals(otherValue);
		}
	}

	private static String deCamelize(String fieldName) {
		StringBuilder buf = new StringBuilder();
		for (int i = 0; i < fieldName.length(); i++) {
			char c = fieldName.charAt(i);
			if (i == 0) {
				c = toUpperCase(c);
			} else if (isUpperCase(c)) {
				buf.append(" ");
			}
			buf.append(c);

		}
		return buf.toString();
	}

    @SuppressWarnings("unchecked")
    void validate(final ConfigurationItem ci, final List<ValidationMessage> messages) {
		ValidationContext context = new ValidationContext() {
			@Override
			public void error(String message, Object... params) {
				messages.add(new ValidationMessage(ci.getId(), name, format(message, params)));
			}
		};
		for (Validator<?> validationRule : validationRules) {
			((Validator<Object>) validationRule).validate(get(ci), context);
		}
	}

	public boolean isInspectionProperty() {
		return inspectionProperty;
	}

	public boolean isRequiredForInspection() {
		return requiredForInspection;
	}

	public boolean isTransient() {
		return isTransient;
	}

	public String getFqn() {
		return declaringDescriptor.getType() + "." + name;
	}

	@Override
	public String toString() {
		return getFqn();
	}

	private static final Logger logger = LoggerFactory.getLogger(PropertyDescriptor.class);

	public Object emptyValue() {
		switch (kind) {
		case BOOLEAN:
			return false;
		case SET_OF_STRING:
		case SET_OF_CI:
			return newHashSet();
		case LIST_OF_STRING:
		case LIST_OF_CI:
			return newArrayList();
		case MAP_STRING_STRING:
			return newHashMap();
		default:
			return null;
		}
	}

	/*
	 * All the setters you could ever want, and more.
	 */

	public void setDeclaringDescriptor(Descriptor declaringDescriptor) {
    	this.declaringDescriptor = declaringDescriptor;
    }

	public void setName(String name) {
    	this.name = name;
    }

	public void setField(Field field) {
    	this.field = field;
    }

	public void setAsContainment(boolean asContainment) {
    	this.asContainment = asContainment;
    }

	public void setCategory(String category) {
    	this.category = category;
    }

	public void setDescription(String description) {
    	this.description = description;
    }

	public void setLabel(String label) {
    	this.label = label;
    }

	public void setPassword(boolean password) {
    	this.password = password;
    }

	public void setRequired(boolean required) {
    	this.required = required;
    }

	public void setSize(Property.Size size) {
    	this.size = size;
    }

	public void setKind(PropertyKind kind) {
    	this.kind = kind;
    }

	public void setEnumValues(List<String> enumValues) {
    	this.enumValues = enumValues;
    }

	public void setEnumClass(Class<?> enumClass) {
    	this.enumClass = enumClass;
    }

	public void setReferencedType(Type referencedType) {
    	this.referencedType = referencedType;
    }

	public void setHidden(boolean hidden) {
    	this.hidden = hidden;
    }

	public void setInspectionProperty(boolean inspectionProperty) {
    	this.inspectionProperty = inspectionProperty;
    }

	public void setRequiredForInspection(boolean requiredForInspection) {
    	this.requiredForInspection = requiredForInspection;
    }

	public void setTransient(boolean isTransient) {
    	this.isTransient = isTransient;
    }

	public void setValidationRules(Set<Validator<?>> validationRules) {
    	this.validationRules = validationRules;
    }
	
	public void setDefaultValue(String defaultValue) {
		GlobalContext.update(this, defaultValue);
	}
}
