package com.xebialabs.deployit.test.deployment;

import static com.google.common.collect.Maps.newHashMap;
import static java.lang.String.format;

import java.util.List;
import java.util.Map;

import com.xebialabs.deployit.deployment.planner.DeploymentPlanner;
import com.xebialabs.deployit.deployment.planner.DeploymentPlannerFactory;
import com.xebialabs.deployit.inspection.Inspector;
import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentExecutionContext;
import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentStep;
import com.xebialabs.deployit.plugin.api.deployment.execution.Plan;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.execution.ExecutionContext;
import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.test.support.LoggingDeploymentExecutionContext;
import com.xebialabs.deployit.test.support.LoggingExecutionContext;

public class DeployitTester {

    private final DeploymentPlanner planner;
	private final Inspector inspector;
	private final InMemoryRepository repository;
	
    public static DeployitTester build() {
        DeploymentPlannerFactory plannerFactory = new DeploymentPlannerFactory();
	    InMemoryRepository inMemoryRepository = new InMemoryRepository();
	    return new DeployitTester(plannerFactory.planner(inMemoryRepository), new Inspector(), inMemoryRepository);
    }

    private DeployitTester(DeploymentPlanner planner, Inspector inspector, InMemoryRepository repository) {
        this.planner = planner;
	    this.inspector = inspector;
	    this.repository = repository;
    }

    public Plan resolvePlan(DeltaSpecification spec) {
        return planner.plan(spec);
    }

	public List<ConfigurationItem> inspect(ConfigurationItem item) {
		LoggingExecutionContext ctx = new LoggingExecutionContext(DeployitTester.class);
		try {
			return inspect(item, ctx);
		} finally {
			ctx.destroy();
		}
	}

	public List<ConfigurationItem> inspect(ConfigurationItem item, ExecutionContext ctx) {
	    return inspector.inspect(item, ctx);
    }

    @SuppressWarnings("rawtypes")
    public Deployed generateDeployed(Deployable d, Container c, Type deployedType) {
    	return generateDeployed(d, c, deployedType, null);
    }
    
	@SuppressWarnings("rawtypes")
    public Deployed generateDeployed(Deployable d, Container c, Type deployedType, Map<String, String> placeholders) {
        Descriptor deployed = DescriptorRegistry.getDescriptor(deployedType);
        Deployed<Deployable, Container> configurationItem = deployed.newInstance();
        configurationItem.setDeployable(d);
        configurationItem.setContainer(c);

        configurationItem.setId(c.getId() + "/" + substringAfterLastSlash(d.getId()));

        Descriptor deployableDescriptor = DescriptorRegistry.getDescriptor(d.getType());
        for (PropertyDescriptor propertyDescriptor : deployed.getPropertyDescriptors()) {
            String name = propertyDescriptor.getName();
            PropertyDescriptor deployablePropertyDescriptor = deployableDescriptor.getPropertyDescriptor(name);
            if (deployablePropertyDescriptor != null) {
            	if(propertyDescriptor.getName().equals("placeholders")) {
            		propertyDescriptor.set(configurationItem, newHashMap());
            	} else {
            		propertyDescriptor.set(configurationItem, deployablePropertyDescriptor.get(d));
            	}
            }
        }
        
        if(configurationItem instanceof DerivedArtifact<?>) {
        	DerivedArtifact<?> da = (DerivedArtifact<?>) configurationItem;
        	if(placeholders != null) {
        		da.setPlaceholders(placeholders);
        	}
        	da.initFile(new SimpleReplacer());
        }

        return configurationItem;
    }

    private String substringAfterLastSlash(String id) {
        int i = id.lastIndexOf('/');
        if (i > -1) {
            return id.substring(i+1);
        }
        return id;
    }

	public Step.Result executePlan(Plan plan) {
		LoggingDeploymentExecutionContext ctx = new LoggingDeploymentExecutionContext(getClass());
		try {
			return executePlan(plan, ctx);
		} finally {
			ctx.destroy();
		}
	}

	public Step.Result executePlan(Plan plan, DeploymentExecutionContext context) {
	    Step.Result result = Step.Result.Warn;
	    List<DeploymentStep> steps = plan.getSteps();
	    try {
	    	for (DeploymentStep step : steps) {
	    		result = step.execute(context);
	    		if (result == Step.Result.Fail) {
	    			break;
	    		}
	    	}
	    	return result;
	    } catch (Exception e) {
	    	throw new DeployitTesterException(e);
	    }
    }

	public InMemoryRepository getRepository() {
		return repository;
	}

	@SuppressWarnings("serial")
    public static class DeployitTesterException extends RuntimeException {
        DeployitTesterException(String message, Object... params) {
            super(format(message, params));
        }

        DeployitTesterException(Exception e) {
            super(e);
        }
    }

}

