package com.xebialabs.deployit.plugin.was.step;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;

import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.plugin.was.ci.WasCluster;
import com.xebialabs.deployit.plugin.was.ci.WasNode;
import com.xebialabs.deployit.util.ExtendedStringUtils;

/**
 * Modifies the properties of a single member (determined by memberNumber) of a {@link WasCluster} on a {@link WasNode}
 */
@SuppressWarnings("serial")
public class ModifyWasClusterMemberStep extends WasStepBase {

	protected static final String MODIFY_CLUSTER_MEMBER_SCRIPT_RESOURCE_PATH = STEP_RESOURCES_PATH + "configure-cluster-member.py";

	protected WasCluster cluster;

	protected WasNode node;

	protected int memberNumber;

	public ModifyWasClusterMemberStep(WasCluster cluster, WasNode node, int memberNumber) {
		super(cluster.getCell());
		this.cluster = cluster;
		this.node = node;
		this.memberNumber = memberNumber;
		setDescription("Configure member " + memberNumber + " of cluster " + ExtendedStringUtils.getQuotedString(cluster.getLabel()) + " on node "
				+ ExtendedStringUtils.getQuotedString(node.getLabel()) + " in cell " + ExtendedStringUtils.getQuotedString(cell.getLabel()));
	}

	public boolean execute(StepExecutionContext ctx) {
		String clusterMemberNamesPrefix = WasCluster.getMemberNamesPrefix(cluster.getName(), node.getName());
		String serverName = clusterMemberNamesPrefix + memberNumber;
		return (executeWsadminJythonScriptResource(ctx, MODIFY_CLUSTER_MEMBER_SCRIPT_RESOURCE_PATH, getArgs(node.getName(), serverName).toArray(new String[] {})) == 0);
	}

	private List<String> getArgs(String nodeName, String serverName) {

		List<String> args = new ArrayList<String>();
		args.add(cell.getName());
		args.add(nodeName);
		args.add(serverName);

		addConfigureServerArgument("classpath", cluster.getClasspath(), args);
		addConfigureServerArgument("bootclasspath", cluster.getBootClasspath(), args);
		addConfigureServerArgument("initheapsize", cluster.getInitHeapSize(), args);
		addConfigureServerArgument("maxheapsize", cluster.getMaxHeapSize(), args);
		addConfigureServerArgument("jvmargs", cluster.getJvmArguments(), args);
		addConfigureServerArgument("disablejit", Boolean.toString(cluster.getDisableJit()), args);

		if (cluster.getApplicationClassLoaderPolicyAndMode() != null) {
			switch (cluster.getApplicationClassLoaderPolicyAndMode()) {
			case MULTIPLE:
				args.add("applicationclassloaderpolicy");
				args.add("MULTIPLE");
				break;
			case SINGLE_PARENT_FIRST:
				args.add("applicationclassloaderpolicy");
				args.add("SINGLE");
				args.add("applicationclassloadingmode");
				args.add("PARENT_FIRST");
				break;
			case SINGLE_PARENT_LAST:
				args.add("applicationclassloaderpolicy");
				args.add("SINGLE");
				args.add("applicationclassloadingmode");
				args.add("PARENT_LAST");
				break;
			}
		}

		addConfigureServerArgument("workdir", cluster.getWorkingDir(), args);
		addConfigureServerArgument("umask", cluster.getUmask(), args);

		addConfigureServerArgument("stdoutlog", getLogPathForMember(serverName, cluster.getStdOut()), args);
		addConfigureServerArgument("stderrlog", getLogPathForMember(serverName, cluster.getStdErr()), args);
		addConfigureServerArgument("jvmstdoutlog", getLogPathForMember(serverName, cluster.getJvmStdOut()), args);
		addConfigureServerArgument("jvmstderrlog", getLogPathForMember(serverName, cluster.getJvmStdErr()), args);

		addConfigureServerArgument("servletcaching", Boolean.toString(cluster.getServletCaching()), args);
		addConfigureServerArgument("sessmaxinmem", cluster.getMaximumSessionsInMemory(), args);
		addConfigureServerArgument("sesstimeout", cluster.getSessionTimeout(), args);
		addConfigureServerArgument("sesscookieenabled", Boolean.toString(cluster.getEnableSessionCookies()), args);
		if (StringUtils.isNotBlank(cluster.getCookieName())) {
			args.add("sesscookie");
			args.add(StringUtils.defaultIfEmpty(cluster.getCookieName(), "JSESSIONID"));
			args.add(StringUtils.defaultString(cluster.getCookieDomain()));
			args.add(StringUtils.defaultIfEmpty(cluster.getCookiePath(), "/"));
		}

		return args;

	}

	private static String getLogPathForMember(String serverName, String logPrefix) {
		if (StringUtils.isBlank(logPrefix)) {
			return logPrefix;
		} else {
			return logPrefix + serverName + ".txt";
		}
	}

	private static void addConfigureServerArgument(String propname, String propvalue, String suffix, List<String> args) {
		if (StringUtils.isNotBlank(propvalue)) {
			args.add(propname);
			if (suffix != null) {
				args.add(propvalue + suffix);
			} else {
				args.add(propvalue);
			}
		}
	}

	private static void addConfigureServerArgument(String propname, String propvalue, List<String> args) {
		addConfigureServerArgument(propname, propvalue, null, args);
	}

	private static void addConfigureServerArgument(String propname, int propvalue, List<String> args) {
		if (propvalue != 0) {
			args.add(propname);
			args.add(Integer.toString(propvalue));
		}
	}

}
