/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.was.ci;

import static com.xebialabs.deployit.hostsession.HostFileUtils.putStringToHostFile;
import static com.xebialabs.deployit.plugin.was.utils.RegexUtils.regexMatch;
import static org.apache.commons.io.FilenameUtils.getBaseName;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.core.io.ClassPathResource;

import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.StepExecutionContextCallbackHandler;
import com.xebialabs.deployit.StepExecutionContextListener;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.CommandExecution;
import com.xebialabs.deployit.hostsession.CommandExecutionCallbackHandler;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.plugin.was.step.WasStepBase;

public class WasCellConfigurationExecutor {

	public static final String GET_OBJECT_BY_CONTAINMENT_PATH_SCRIPT = WasStepBase.STEP_RESOURCES_PATH + "get-object-by-containment-path.py";

	public static final String GET_OBJECT_BY_CONTAINMENT_PATH_AND_SUBLEVEL_TYPE_SCRIPT = WasStepBase.STEP_RESOURCES_PATH
	        + "get-object-by-containment-path-and-sublevel-type.py";

	public static final String GET_OBJECT_BY_CONTAINMENT_PATH_AND_SUBLEVEL_TYPE_SCRIPT_NAMES_ONLY = WasStepBase.STEP_RESOURCES_PATH
	        + "get-object-by-containment-path-and-sublevel-type-names-only.py";

	public static final String GET_OBJECT_BY_CONTAINMENT_PATH_AND_SUBLEVEL_NAME_SCRIPT = WasStepBase.STEP_RESOURCES_PATH
	        + "get-object-by-containment-path-and-sublevel-name.py";

	public static final String GET_OBJECT_BY_CONTAINMENT_PATH_AND_SUBLEVEL_TYPE_AND_NAME_SCRIPT = WasStepBase.STEP_RESOURCES_PATH
	        + "get-object-by-containment-path-and-sublevel-type-and-name.py";

	public static final String GET_CELL_PROPERTIES_PATH_SCRIPT = WasStepBase.STEP_RESOURCES_PATH + "get-cell-properties.py";

	/**
	 * The system property to set to disable the usage of the wsadmin daemon.
	 */
	public static final String DISABLE_DAEMON = "com.xebialabs.deployit.plugin.was.disableDaemon";

	private static final String DAEMON_STARTED = "DEPLOYIT-DAEMON-STARTED";

	private static final String DAEMON_EXIT_CODE_MARKER = "DEPLOYIT-DAEMON-EXIT-VALUE: ";

	private static final String DAEMON_END_OF_COMMAND_MARKER = "DEPLOYIT-DAEMON-END-OF-COMMAND";

	private static final String CONFIG_DIR_NAME = "config";

	private static final String PLUGIN_FILENAME = "plugin-cfg.xml";

	private WasCell cell;

	public WasCellConfigurationExecutor(WasCell cell) {
		this.cell = cell;
	}

	public Host getCellHost() {
		return cell.getCellHost();
	}

	public String getConfigDirPath() {
		String fileSep = getCellHost().getOperatingSystemFamily().getFileSeparator();
		return cell.getWasHome() + fileSep + CONFIG_DIR_NAME;
	}

	public String getWsadminPath() {
		return getCommandPath("wsadmin");
	}

	public String getVersionInfoPath() {
		return getCommandPath("versionInfo");
	}

	public String getStartServerPath() {
		return getCommandPath("startServer");
	}

	public String getStopServerPath() {
		return getCommandPath("stopServer");
	}

	public String getServerStatusPath() {
		return getCommandPath("serverStatus");
	}

	private String getCommandPath(String command) {
		String fileSep = getCellHost().getOperatingSystemFamily().getFileSeparator();
		return cell.getWasHome() + fileSep + "bin" + fileSep + command + getCellHost().getOperatingSystemFamily().getScriptExtension();
	}

	public String getPluginFilePath() {
		String fileSep = getCellHost().getOperatingSystemFamily().getFileSeparator();
		return getConfigDirPath() + fileSep + "cells" + fileSep + PLUGIN_FILENAME;
	}

	public HostSession connectToAdminHost() {
		return getCellHost().getHostSession();
	}

	public int executeWsadminJythonScript(StepExecutionContext ctx, HostSession cellHostSession, String scriptPath, String... args) {
		HostFile invokingPyFile = createInvokingPyFile(cellHostSession, scriptPath, args);
		String disableDaemon = System.getProperty(DISABLE_DAEMON);
		if (disableDaemon == null) {
			return executeWsadminJythonScriptWithDaemon(ctx, cellHostSession, invokingPyFile);
		} else {
			return executeWsadminJythonScriptDirectly(new StepExecutionContextCallbackHandler(ctx), cellHostSession, invokingPyFile);
		}
	}

	private HostFile createInvokingPyFile(HostSession cellHostSession, String scriptPath, String... args) {
		StringBuffer invokingPyContents = new StringBuffer();
		invokingPyContents.append("import sys;\r\nsys.argv=[");
		boolean first = true;
		for (String arg : args) {
			if (!first) {
				invokingPyContents.append(",");
			} else {
				first = false;
			}
			invokingPyContents.append("\'");
			invokingPyContents.append(arg);
			invokingPyContents.append("\'");
		}
		invokingPyContents.append("];\r\nexecfile(\'");
		invokingPyContents.append(scriptPath);
		invokingPyContents.append("\')\r\n");

		HostFile invokingPyFile = cellHostSession.getTempFile(getBaseName(scriptPath), "-invoker.py");
		putStringToHostFile(invokingPyContents.toString(), invokingPyFile);
		return invokingPyFile;
	}

	public int executeWsadminJythonScriptWithDaemon(final StepExecutionContext ctx, HostSession cellHostSession, HostFile invokingPyFile) {
		try {
			CommandExecution wsadminExecution = getWsadminDaemonCommandExecution(ctx);
			if (wsadminExecution == null) {
				logger.warn("Cannot create wsadmin daemon session, executing script directly");
				return executeWsadminJythonScriptDirectly(new StepExecutionContextCallbackHandler(ctx), cellHostSession, invokingPyFile);
			}

			String daemonLine = "execfile(\'" + invokingPyFile.getPath() + "\')\r\n";
			OutputStream stdin = wsadminExecution.getStdin();
			stdin.write(daemonLine.getBytes());
			stdin.flush();

			final BufferedReader stdout = new BufferedReader(new InputStreamReader(wsadminExecution.getStdout()));
			final BufferedReader stderr = new BufferedReader(new InputStreamReader(wsadminExecution.getStderr()));
			Thread stderrThread = new Thread(new Runnable() {
				public void run() {
					try {
						while (!Thread.interrupted()) {
							String stderrLine = stderr.readLine();
							if (stderrLine == null) {
								break;
							}
							ctx.logError(stderrLine);
						}
					} catch (InterruptedIOException ie) {
						// ignore, is expected
					} catch (IOException exc) {
						exc.printStackTrace();
					}
				}
			}, "stderr printer for wsadmin daemon running on " + this);
			stderrThread.start();
			try {
				int exitCode = 0;
				for (;;) {
					String stdoutLine = stdout.readLine();
					if (stdoutLine == null) {
						throw new RuntimeIOException("Cannot execute script " + invokingPyFile.getPath() + " on " + getCellHost()
						        + ": lost connection to the wsadmin daemon");
					}

					if (stdoutLine.startsWith(DAEMON_EXIT_CODE_MARKER)) {
						try {
							exitCode = Integer.parseInt(stdoutLine.substring(DAEMON_EXIT_CODE_MARKER.length()));
						} catch (NumberFormatException ignored) {
						}
					} else if (stdoutLine.equals(DAEMON_END_OF_COMMAND_MARKER)) {
						break;
					} else {
						ctx.logOutput(stdoutLine);
					}
				}

				return exitCode;
			} finally {
				stderrThread.interrupt();
			}
		} catch (IOException exc) {
			throw new RuntimeIOException("Cannot execute script " + invokingPyFile.getPath() + " on " + getCellHost(), exc);
		}
	}

	private CommandExecution getWsadminDaemonCommandExecution(StepExecutionContext context) {
		String key = "CELL_SESSION_" + cell;
		WsadminDaemonExecution wrappedExecution = (WsadminDaemonExecution) context.getAttribute(key);
		if (wrappedExecution == null) {
			HostSession daemonSession = connectToAdminHost();
			HostFile uploadedDaemon = daemonSession.copyToTemporaryFile(new ClassPathResource(WasStepBase.STEP_RESOURCES_PATH + "daemon.py"));
			String[] cmdArray = buildWsadminCommandLine(uploadedDaemon);

			CommandExecution daemonExecution = daemonSession.startExecute(cmdArray);
			if (daemonExecution == null) {
				daemonSession.close();
				return null;
			}
			final BufferedReader stdout = new BufferedReader(new InputStreamReader(daemonExecution.getStdout()));
			for (;;) {
				String stdoutLine;
				try {
					stdoutLine = stdout.readLine();
					if (stdoutLine == null) {
						throw new RuntimeIOException("Cannot start wsadmin daemon: lost connection to the wsadmin daemon");
					}
					context.logOutput(stdoutLine);
					if (stdoutLine.startsWith(DAEMON_STARTED)) {
						break;
					}
				} catch (IOException exc) {
					throw new RuntimeIOException("Cannot start wsadmin daemon", exc);
				}
			}

			wrappedExecution = new WsadminDaemonExecution(daemonSession, daemonExecution);
			context.setAttribute(key, wrappedExecution);
		}
		return wrappedExecution.getCommandExecution();
	}

	public static final class WsadminDaemonExecution implements StepExecutionContextListener {

		private HostSession session;

		private CommandExecution execution;

		public WsadminDaemonExecution(HostSession session, CommandExecution execution) {
			this.session = session;
			this.execution = execution;
		}

		public CommandExecution getCommandExecution() {
			return execution;
		}

		public void contextDestroyed() {
			OutputStream stdin = execution.getStdin();
			try {
				stdin.write("QUIT\r\n".getBytes());
				stdin.flush();
				execution.waitFor();
			} catch (IOException exc) {
				logger.error("Error stopping wsadmin daemon", exc);
			} catch (RuntimeException exc) {
				logger.error("Error stopping wsadmin daemon", exc);
			}

			session.close();
		}

	}

	public int executeWsadminJythonScriptDirectly(CommandExecutionCallbackHandler handler, HostSession s, HostFile invokingPyFile) {
		String[] cmdArray = buildWsadminCommandLine(invokingPyFile);

		return s.execute(handler, cmdArray);
	}

	private String[] buildWsadminCommandLine(HostFile pyFile) {
		List<String> cmd = new ArrayList<String>();
		cmd.add(getWsadminPath());
		if (StringUtils.isNotBlank(cell.getUsername())) {
			cmd.add("-user");
			cmd.add(cell.getUsername());
			if (!StringUtils.isEmpty(cell.getPassword())) {
				cmd.add("-password");
				cmd.add(cell.getPassword());
			}
		}
		if (cell.getPort() != 0) {
			cmd.add("-port");
			cmd.add(Integer.toString(cell.getPort()));
		}
		cmd.add("-lang");
		cmd.add("jython");
		cmd.add("-f");
		cmd.add(pyFile.getPath());
		return (String[]) cmd.toArray(new String[cmd.size()]);
	}

	public Map<String, String> getCellPropertiesInfo(StepExecutionContext ctx) {
		return getObjectInfo(ctx, null, null, null, false);
	}

	public Map<String, String> getObjectInfo(StepExecutionContext ctx, String objectContainmentPath) {
		return getObjectInfo(ctx, objectContainmentPath, null, null, false);
	}

	public Map<String, String> getObjectInfoWithSubLevelByType(StepExecutionContext ctx, String objectContainmentPath, String subLevelType) {
		return getObjectInfo(ctx, objectContainmentPath, subLevelType, null, false);
	}

	public Collection<String> getObjectInfoWithSubLevelByTypeNamesOnly(StepExecutionContext ctx, String objectContainmentPath, String subLevelType) {
		Map<String, String> map = getObjectInfo(ctx, objectContainmentPath, subLevelType, null, true);
		if (map == null) {
			return null;
		} else {
			return map.values();
		}
	}

	public Map<String, String> getObjectInfoWithSubLevelByName(StepExecutionContext ctx, String objectContainmentPath, String subLevelName) {
		return getObjectInfo(ctx, objectContainmentPath, null, subLevelName, false);
	}

	public Map<String, String> getObjectInfoWithSubLevelByTypeAndName(StepExecutionContext ctx, String objectContainmentPath, String subLevelType,
	        String subLevelName) {
		return getObjectInfo(ctx, objectContainmentPath, subLevelType, subLevelName, false);
	}

	private Map<String, String> getObjectInfo(final StepExecutionContext ctx, String objectContainmentPath, String subLevelType, String subLevelName,
	        boolean namesOnly) {
		final Map<String, String> mapped = new HashMap<String, String>();
		StepExecutionContext capturingPassthruContext = new StepExecutionContext() {

			private int nodes = 0;
			private int clusters = 0;
			private int servers = 0;

			public void logOutput(String line) {
				ctx.logOutput(line);
				int offset = 0;
				String[] args = regexMatch(line, "\\[(\\w+)\\ (.*)\\]");
				if (args != null) {
					String[] betweenQuotes = regexMatch(args[1], "\"(.*)\"");
					if (betweenQuotes != null) {
						mapped.put(args[0], betweenQuotes[0]);
					} else {
						if (!args[1].equals("[]")) {
							mapped.put(args[0], args[1]);
						} else {
							mapped.put(args[0], "");
						}
					}
				} else if (line.indexOf("|node.xml#") != -1) {
					nodes++;
					if (line.startsWith("\"") && line.endsWith("\"")) {
						offset = 1;
					}
					mapped.put("node_" + nodes, line.substring(offset, line.indexOf('(')));
				} else if (line.indexOf("|server.xml#") != -1) {
					servers++;
					if (line.startsWith("\"") && line.endsWith("\"")) {
						offset = 1;
					}
					mapped.put("server_" + servers, line.substring(offset, line.indexOf('(')));
				} else if (line.indexOf("|cluster.xml#") != -1) {
					if (line.startsWith("\"") && line.endsWith("\"")) {
						offset = 1;
					}
					clusters++;
					mapped.put("cluster_" + clusters, line.substring(offset, line.indexOf('(')));
				}
			}

			public void logError(String line) {
				ctx.logError(line);
			}

			public void logError(String error, Throwable t) {
				ctx.logError(error, t);

			}

			public Object getAttribute(String name) {
				return ctx.getAttribute(name);
			}

			public void setAttribute(String name, Object object) {
				ctx.setAttribute(name, object);
			}

		};

		HostSession s = connectToAdminHost();
		try {
			String[] args;
			String scriptResourcePath;
			if (subLevelType == null) {
				if (subLevelName == null) {
					if (objectContainmentPath == null) {
						scriptResourcePath = GET_CELL_PROPERTIES_PATH_SCRIPT;
						args = new String[] {};
					} else {
						scriptResourcePath = GET_OBJECT_BY_CONTAINMENT_PATH_SCRIPT;
						args = new String[] { objectContainmentPath };
					}
				} else {
					scriptResourcePath = GET_OBJECT_BY_CONTAINMENT_PATH_AND_SUBLEVEL_NAME_SCRIPT;
					args = new String[] { objectContainmentPath, subLevelName };
				}
			} else {
				if (subLevelName == null) {
					if (namesOnly) {
						scriptResourcePath = GET_OBJECT_BY_CONTAINMENT_PATH_AND_SUBLEVEL_TYPE_SCRIPT_NAMES_ONLY;
					} else {
						scriptResourcePath = GET_OBJECT_BY_CONTAINMENT_PATH_AND_SUBLEVEL_TYPE_SCRIPT;
					}
					args = new String[] { objectContainmentPath, subLevelType };
				} else {
					scriptResourcePath = GET_OBJECT_BY_CONTAINMENT_PATH_AND_SUBLEVEL_TYPE_AND_NAME_SCRIPT;
					args = new String[] { objectContainmentPath, subLevelType, subLevelName };
				}
			}

			String scriptPath = s.copyToTemporaryFile(new ClassPathResource(scriptResourcePath)).getPath();
			int res = executeWsadminJythonScript(capturingPassthruContext, s, scriptPath, args);

			if (res != 0) {
				return null;
			} else {
				return mapped;
			}
		} finally {
			s.close();
		}
	}

	private static Logger logger = Logger.getLogger(WasCellConfigurationExecutor.class);

}
