/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.was.ci;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;

import com.xebialabs.deployit.BaseConfigurationItem;
import com.xebialabs.deployit.ChangePlan;
import com.xebialabs.deployit.ConfigurationItem;
import com.xebialabs.deployit.ConfigurationItemProperty;
import com.xebialabs.deployit.ConfigurationItemRoot;
import com.xebialabs.deployit.ConfigurationItemProperty.Size;
import com.xebialabs.deployit.Discoverable;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.plugin.was.step.CheckServerIsInClusterStep;
import com.xebialabs.deployit.plugin.was.step.CheckServerIsOnNodeStep;
import com.xebialabs.deployit.plugin.was.step.CheckWasAdminAvailableStep;
import com.xebialabs.deployit.plugin.was.step.RetrieveWasCellNameStep;
import com.xebialabs.deployit.plugin.was.step.RetrieveWasSubtypesStep;
import com.xebialabs.deployit.plugin.was.step.RetrieveWasVersionStep;
import com.xebialabs.deployit.step.CheckDirExistsStep;
import com.xebialabs.deployit.step.CheckFileExistenceStep;
import com.xebialabs.deployit.step.HostConnectionStep;
import com.xebialabs.deployit.util.LoggingStepExecutionContext;

/**
 * A WAS deployment manager.
 */
@SuppressWarnings("serial")
@ConfigurationItem(root = ConfigurationItemRoot.INFRASTRUCTURE, discoverable = true, category = "middleware", description = "A WebSphere Application Server deployment manager (WAS ND)")
public class WasDeploymentManager extends BaseConfigurationItem implements WasCell, Discoverable<WasDeploymentManager> {

	@ConfigurationItemProperty(required = true, label = "Host", description = "Host on which the WAS deployment manager runs")
	private Host host;

	@ConfigurationItemProperty(required = true, identifying = true, label = "WebSphere name", description = "Name of the WebSphere cell, e.g. MyCell, WASCell, Cell01")
	private String name;

	@ConfigurationItemProperty(required = true, discoveryParam = true, label = "Deployment manager profile path", description = "Root path of the WebSphere deployment manager profile. e.g. /opt/ws/6.1/profiles/dmgr", size = Size.LARGE)
	private String wasHome;

	@ConfigurationItemProperty(required = false, label = "Administrative port", description = "TCP port which is used to login to the WebSphere deployment manager, default is 8879")
	private int port;

	@ConfigurationItemProperty(required = false, discoveryParam = true, discoveryRequired = false, label = "Administrative username", description = "Username which is used to login to the WebSphere deployment manager")
	private String username;

	@ConfigurationItemProperty(required = false, password = true, discoveryParam = true, discoveryRequired = false, label = "Administrative password", description = "Password which is used to login to the WebSphere deployment manager")
	private String password;

	@ConfigurationItemProperty(required = true, label = "WAS version", description = "Version of WebSphere Application Server")
	private WasVersion version;

	protected transient WasCellConfigurationExecutor executor;

	private WasCellConfigurationExecutor getWasCellConfigurationExecutor() {
		if (executor == null) {
			executor = new WasCellConfigurationExecutor(this);
		}
		return executor;
	}

	public Host getHost() {
		return host;
	}

	public void setHost(Host host) {
		this.host = host;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getCellName() {
		return name;
	}

	public void setCellName(String cellName) {
		this.name = cellName;
	}

	public int getPort() {
		return port;
	}

	public void setPort(int port) {
		this.port = port;
	}

	public String getUsername() {
		return username;
	}

	public void setUsername(String username) {
		this.username = username;
	}

	public String getPassword() {
		return password;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	public String getWasHome() {
		return wasHome;
	}

	public void setWasHome(String wasHome) {
		this.wasHome = wasHome;
	}

	public WasVersion getVersion() {
		return version;
	}

	public void setVersion(WasVersion version) {
		this.version = version;
	}

	public WasDeploymentManager discover(Map<String, Object> discoveredAdditionalInfo, ChangePlan cp) {
		List<Step> steps = getCheckSteps();
		cp.addSteps(steps);
		cp.execute(discoveredAdditionalInfo);
		return this;
	}

	@SuppressWarnings("unchecked")
	public WasDeploymentManager discoverChildrenInfo(Map<String, Object> discoveredAdditionalInfo, ChangePlan cp) {

		LoggingStepExecutionContext context = new LoggingStepExecutionContext(logger);
		try {
			List<WasNodeAgent> discoveredNodes = new ArrayList<WasNodeAgent>();
			Object children = discoveredAdditionalInfo.get("WasNode");
			if (children != null) {
				for (String nodeName : (Collection<String>) children) {
					WasNodeAgent discoveredNode = new WasNodeAgent();
					discoveredNode.setCell(this);
					discoveredNode.setName(nodeName);
					discoveredNode.setLabel("Node " + nodeName + " on " + this.getName());
					discoveredNodes.add(discoveredNode);
				}
			}
			discoveredAdditionalInfo.put("1_com.xebialabs.deployit.plugin.was.ci.WasNodeAgent", discoveredNodes);
			discoveredAdditionalInfo.remove("WasNode");
			List<WasManagedServer> discoveredServers = new ArrayList<WasManagedServer>();
			children = discoveredAdditionalInfo.get("WasServer");
			if (children != null) {
				for (String serverName : (Collection<String>) children) {
					WasManagedServer discoveredServer = new WasManagedServer();
					discoveredServer.setName(serverName);
					discoveredServer.setNode(getNodeForServer(serverName, discoveredNodes, context));
					String type = getApplicationServerType(discoveredServer, context);
					if (type != null && type.equals("APPLICATION_SERVER")) {
						discoveredServer.setLabel("Server " + serverName + " on " + discoveredServer.getNode().getLabel());
						discoveredServer = discoveredServer.discover(discoveredAdditionalInfo, cp);
						discoveredServers.add(discoveredServer);
					} else if (type != null && type.equals("WEB_SERVER")) {
						logger.info("Not discovering a web server");
					}
				}
			}
			discoveredAdditionalInfo.put("2_com.xebialabs.deployit.plugin.was.ci.WasManagedServer", discoveredServers);
            discoveredAdditionalInfo.remove("WasServer");
			List<WasCluster> discoveredClusters = new ArrayList<WasCluster>();
			children = discoveredAdditionalInfo.get("WasCluster");
			if (children != null) {
				for (String clusterName : (Collection<String>) children) {
					WasCluster discoveredCluster = new WasCluster();
					discoveredCluster.setName(clusterName);
					discoveredCluster.setCell(this);
					discoveredCluster.setLabel("Cluster " + discoveredCluster.getName() + " on " + this.getName());
					discoveredCluster = discoveredCluster.discover(discoveredAdditionalInfo, cp); // TODO implement that
					// method!
					discoveredCluster.setServers(getServersForCluster(clusterName, discoveredServers, context));
					discoveredClusters.add(discoveredCluster);
				}
			}
			discoveredAdditionalInfo.put("4_com.xebialabs.deployit.plugin.was.ci.WasCluster", discoveredClusters);
			discoveredAdditionalInfo.remove("WasCluster");
			return this;
		} finally {
			context.destroy();
		}
	}

	private String getApplicationServerType(WasManagedServer server, StepExecutionContext context) {
		context.logOutput("Retrieving type of server " + server);
		Map<String, String> serverProperties = getObjectInfo(context, "/Cell:" + server.getNode().getCell().getName() + "/Node:" + server.getNode().getName()
		        + "/Server:" + server.getName());
		if (serverProperties == null) {
			logger.error("Did not received server properties for : " + "/Cell:" + server.getNode().getCell().getName() + "/Node:" + server.getNode().getName()
			        + "/Server:" + server.getName());
		}
		return serverProperties.get("serverType");
	}

	private WasNodeAgent getNodeForServer(String serverName, List<WasNodeAgent> discoveredNodes, StepExecutionContext context) {
		for (WasNodeAgent node : discoveredNodes) {
			context.logOutput("Checking server " + serverName + " is on node " + node);
			Step checkServerIsOnNode = new CheckServerIsOnNodeStep(serverName, node);
			if (checkServerIsOnNode.execute(context)) {
				return node;
			}
		}
		return null;
	}

	private Set<WasManagedServer> getServersForCluster(String clusterName, List<WasManagedServer> discoveredServers, LoggingStepExecutionContext context) {
		Set<WasManagedServer> serversInThisCluster = new HashSet<WasManagedServer>();
		context.logOutput("Retrieving servers " + " in cluster " + clusterName);
		for (WasManagedServer server : discoveredServers) {
			// FIXME: this step should be added to the ChangePlan and then the changeplan should be executed!
			Step checkServerIsInCluster = new CheckServerIsInClusterStep(clusterName, server);
			boolean serverIsInCluster = checkServerIsInCluster.execute(context);
			if (serverIsInCluster) {
				serversInThisCluster.add(server);
			}
		}
		return serversInThisCluster;
	}

	private List<Step> getCheckSteps() {
		List<Step> steps = new ArrayList<Step>();
		steps.add(new HostConnectionStep(getHost()));
		steps.add(new CheckDirExistsStep(getHost(), getWasHome()));
		steps.add(new CheckDirExistsStep(getHost(), getConfigDirPath()));
		steps.add(new CheckFileExistenceStep(getHost(), getVersionInfoPath(), true, false, true));
		steps.add(new RetrieveWasVersionStep(this));
		steps.add(new CheckFileExistenceStep(getHost(), getWsadminPath(), true, false, true));
		steps.add(new CheckWasAdminAvailableStep(this));
		steps.add(new RetrieveWasCellNameStep(this));
		steps.add(new RetrieveWasSubtypesStep(this));
		return steps;
	}

	public Host getCellHost() {
		return host;
	}

	public HostSession connectToAdminHost() {
		return getWasCellConfigurationExecutor().connectToAdminHost();
	}

	public int executeWsadminJythonScript(StepExecutionContext ctx, HostSession cellHostSession, String scriptPath, String... args) {
		return executor.executeWsadminJythonScript(ctx, cellHostSession, scriptPath, args);
	}

	public Map<String, String> getCellPropertiesInfo(StepExecutionContext ctx) {
		return getWasCellConfigurationExecutor().getCellPropertiesInfo(ctx);
	}

	public Map<String, String> getObjectInfo(StepExecutionContext ctx, String objectContainmentPath) {
		return getWasCellConfigurationExecutor().getObjectInfo(ctx, objectContainmentPath);
	}

	public Map<String, String> getObjectInfoWithSubLevelByName(StepExecutionContext ctx, String objectContainmentPath, String subLevelName) {
		return getWasCellConfigurationExecutor().getObjectInfoWithSubLevelByName(ctx, objectContainmentPath, subLevelName);
	}

	public Map<String, String> getObjectInfoWithSubLevelByType(StepExecutionContext ctx, String objectContainmentPath, String subLevelType) {
		return getWasCellConfigurationExecutor().getObjectInfoWithSubLevelByType(ctx, objectContainmentPath, subLevelType);
	}

	public Map<String, String> getObjectInfoWithSubLevelByTypeAndName(StepExecutionContext ctx, String objectContainmentPath, String subLevelType,
	        String subLevelName) {
		return getWasCellConfigurationExecutor().getObjectInfoWithSubLevelByTypeAndName(ctx, objectContainmentPath, subLevelType, subLevelName);
	}

	public Collection<String> getObjectInfoWithSubLevelByTypeNamesOnly(StepExecutionContext ctx, String objectContainmentPath, String subLevelType) {
		return getWasCellConfigurationExecutor().getObjectInfoWithSubLevelByTypeNamesOnly(ctx, objectContainmentPath, subLevelType);
	}

	public String getConfigDirPath() {
		return getWasCellConfigurationExecutor().getConfigDirPath();
	}

	public String getPluginFilePath() {
		return getWasCellConfigurationExecutor().getPluginFilePath();
	}

	public String getVersionInfoPath() {
		return getWasCellConfigurationExecutor().getVersionInfoPath();
	}

	public String getWsadminPath() {
		return getWasCellConfigurationExecutor().getWsadminPath();
	}

	public String getStartServerPath() {
		return getWasCellConfigurationExecutor().getStartServerPath();
	}

	public String getStopServerPath() {
		return getWasCellConfigurationExecutor().getStopServerPath();
	}

	public String getServerStatusPath() {
		return getWasCellConfigurationExecutor().getServerStatusPath();
	}

	public WasCell getCell() {
		return this;
	}

	public String getShortTypeDescription() {
		return "cell";
	}

	public String getWasConfigIdType() {
		return "Cell";
	}

	public String getWasTargetType() {
		return "Cell";
	}

	private static Logger logger = Logger.getLogger(WasDeploymentManager.class);
}
