/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.was.ci;

import static java.util.Collections.EMPTY_LIST;
import static java.util.Collections.EMPTY_SET;

import java.util.Collections;
import java.util.List;
import java.util.Set;

import com.xebialabs.deployit.ConfigurationItem;
import com.xebialabs.deployit.ConfigurationItemProperty;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.artifact.Ear;
import com.xebialabs.deployit.ci.artifact.mapping.EarMapping;
import com.xebialabs.deployit.plugin.was.utils.WasEarUtils;
import com.xebialabs.deployit.plugin.was.utils.WebSphereNameBuilder;

@SuppressWarnings("serial")
@ConfigurationItem(description = "A mapping of an EAR to a WebSphere target")
public class WasEarMapping extends EarMapping<WasTarget> {

	@ConfigurationItemProperty(description = "Set of webservers that expose the Eneterprise Application")
	private Set<WasManagedApacheHttpdServer> webservers;

	@ConfigurationItemProperty(description = "Set of shared library which will used by the ear", category = "Shared library references")
	private Set<WasSharedLibrary> sharedLibraries;

	@ConfigurationItemProperty(description = "Specifies the order in which applications are started. Lower values start earlier.")
	private int startingWeight = 1;

	@ConfigurationItemProperty(description = "If true, the artifact name will be suffixed with the name of the target.")
	private boolean suffixArtifactNameWithTarget = false;

	@ConfigurationItemProperty(description = "Specifies the Classloader mode", category = "Class loading")
	private WasClassLoaderMode classLoaderMode;

	@ConfigurationItemProperty(description = "Specifies the Classloader policy", category = "Class loading")
	private WasClassLoaderPolicy classLoaderPolicy;

	@ConfigurationItemProperty(description = "Map Security role to users and groups used by EnterPrise Application", category = "Security")
	private List<SecurityRoleUserGroupMappings> securityRoleUserGroupMappings;

	@ConfigurationItemProperty(description = "Specifies the Class loader mode to WARs in EAR", category = "Web modules")
	private List<WasWarClassLoaderMapping> warClassLoaderMapping;

	@ConfigurationItemProperty(description = "Map Wars to Webservers and Virtual hosts in EnterPrise Application", category = "Web modules")
	private List<WarsWebserversVirtualHostMapping> warsWebserversVirtualHostMapping;

	@SuppressWarnings("unchecked")
	public WasEarMapping() {
		// default constructor
		webservers = EMPTY_SET;
		sharedLibraries = EMPTY_SET;
		securityRoleUserGroupMappings = EMPTY_LIST;
		warClassLoaderMapping = EMPTY_LIST;
		warsWebserversVirtualHostMapping = EMPTY_LIST;
	}

	@SuppressWarnings("unchecked")
	public WasEarMapping(Ear source, WasTarget target, String virtualHost) {
		this(source, target, (Set<WasManagedApacheHttpdServer>) EMPTY_SET, virtualHost);
	}

	@SuppressWarnings("unchecked")
	public WasEarMapping(Ear source, WasTarget target, Set<WasManagedApacheHttpdServer> webservers, String virtualHost) {
		this(source, target, webservers, virtualHost, EMPTY_LIST);
	}

	@SuppressWarnings("unchecked")
	public WasEarMapping(Ear source, WasTarget target, Set<WasManagedApacheHttpdServer> webservers, String virtualHost,
	        List<WarsWebserversVirtualHostMapping> warsWebserversVirtualHostMapping) {
		super(source, target, virtualHost);
		this.webservers = webservers;
		this.sharedLibraries = Collections.EMPTY_SET;
		this.securityRoleUserGroupMappings = Collections.EMPTY_LIST;
		this.warsWebserversVirtualHostMapping = warsWebserversVirtualHostMapping;
	}

	@Override
	public void postInit(Deployment deployment) {
		this.setVirtualHost(deployment.getVhostDefinition());
		this.securityRoleUserGroupMappings = WasEarUtils.getSecurityRoleUserGroupMappings(getSource());
		this.warsWebserversVirtualHostMapping = WasEarUtils.getWarsWebserversMapping(getSource());
		this.warClassLoaderMapping = WasEarUtils.getWasWarClassLoaderMappings(getSource());
	}

	/*
	 * The "suffixArtifactNameWithTarget" option means that the WebSphere name for the source should be generated by adding the target to the name property
	 * defined in the CI. Since the EAR is passed as an argument to all the steps with the contract that its name is the WebSphere name under which it should be
	 * deployed, the least dangerous way to achieve this is to return a decorated source that uses this adjusted name.
	 * 
	 * Of course, a better way to address this would be to generate the ultimate WebSphere names on the basis of the wider deployment context, not just by
	 * looking at the name of the source artifact.
	 * 
	 * See DEPLOYIT-315 and DEPLOYIT-376
	 */
	@Override
	public Ear getSource() {
		Ear ear = super.getSource();
		if (!suffixArtifactNameWithTarget) {
			return ear;
		}

		Ear earWithSuffixedName = new Ear();
		earWithSuffixedName.setLabel(ear.getLabel());
		earWithSuffixedName.setLocation(ear.getLocation());
		earWithSuffixedName.setName(WebSphereNameBuilder.getDeployedApplicationName(ear, this));
		return earWithSuffixedName;
	}

	public int getStartingWeight() {
		return startingWeight;
	}

	public void setStartingWeight(int startingWeight) {
		this.startingWeight = startingWeight;
	}

	public boolean isSuffixArtifactNameWithTarget() {
		return suffixArtifactNameWithTarget;
	}

	public Set<WasManagedApacheHttpdServer> getWebservers() {
		return webservers;
	}

	public void setWebservers(Set<WasManagedApacheHttpdServer> webservers) {
		this.webservers = webservers;
	}

	public Set<WasSharedLibrary> getSharedLibraries() {
		return sharedLibraries;
	}

	public void setSharedLibraries(Set<WasSharedLibrary> sharedLibraries) {
		this.sharedLibraries = sharedLibraries;
	}

	public List<WarsWebserversVirtualHostMapping> getWarsWebserversVHMapping() {
		return warsWebserversVirtualHostMapping;
	}

	public void setWarsWebserversVHMapping(List<WarsWebserversVirtualHostMapping> warsWebserversVirtualHostMapping) {
		this.warsWebserversVirtualHostMapping = warsWebserversVirtualHostMapping;
	}

	public WasClassLoaderMode getClassLoaderMode() {
		return classLoaderMode;
	}

	public void setClassLoaderMode(WasClassLoaderMode classLoaderMode) {
		this.classLoaderMode = classLoaderMode;
	}

	public WasClassLoaderPolicy getClassLoaderPolicy() {
		return classLoaderPolicy;
	}

	public void setClassLoaderPolicy(WasClassLoaderPolicy classLoaderPolicy) {
		this.classLoaderPolicy = classLoaderPolicy;
	}

	public List<WasWarClassLoaderMapping> getWarClassLoaderMapping() {
		return warClassLoaderMapping;
	}

	public void setWarClassLoaderMapping(List<WasWarClassLoaderMapping> warClassLoaderMapping) {
		this.warClassLoaderMapping = warClassLoaderMapping;
	}

	public List<SecurityRoleUserGroupMappings> getSecurityRoleUserGroupMappings() {
		return securityRoleUserGroupMappings;
	}

	public void setSecurityRoleUserGroupMappings(List<SecurityRoleUserGroupMappings> securityRoleUserGroupMappings) {
		this.securityRoleUserGroupMappings = securityRoleUserGroupMappings;
	}

	public void setSuffixArtifactNameWithTarget(boolean suffixArtifactNameWithTarget) {
		this.suffixArtifactNameWithTarget = suffixArtifactNameWithTarget;
	}
}
