/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wls.ci;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;

import com.xebialabs.deployit.BaseConfigurationItem;
import com.xebialabs.deployit.ConfigurationItem;
import com.xebialabs.deployit.ConfigurationItemProperty;
import com.xebialabs.deployit.ConfigurationItemProperty.Size;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.hostsession.CommandExecutionCallbackHandler;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.hostsession.HostSessionFactory;

/**
 * A domain in a WebLogic WLS installation.
 * 
 * @see WlsCluster
 * @see WlsServer
 */
@SuppressWarnings("serial")
@ConfigurationItem(description = "A WebLogic Domain.", expandable = true, category = "middleware")
public class WlsDomain extends BaseConfigurationItem {

	@ConfigurationItemProperty(required = true, identifying = true, description = "Name of the WebLogic Domain")
	private String name;

	@ConfigurationItemProperty(required = true, description = "Username which is used to login to the WebLogic Domain.")
	private String username;

	@ConfigurationItemProperty(required = true, password = true, description = "Password which is used to login to the WebLogic Domain.")
	private String password;

	@ConfigurationItemProperty(required = true, label = "WebLogic home", description = "The location of the WebLogic Server installation", size = Size.LARGE)
	private String wlHome;

	@ConfigurationItemProperty(required = true, label = "Java Home", description = "The location of the JDK to use.", size = Size.LARGE)
	private String wlJavaHome;

	@ConfigurationItemProperty(required = false, description = "The location of the WebLogic domain. Defaults to '<WebLogicHome>/../user_projects/domains/<Name>'", size = Size.LARGE)
	private String domainHome;

	@ConfigurationItemProperty(required = true, description = "Port to be used by the AdminServer for this domain", size = Size.SMALL)
	private int port;

	@ConfigurationItemProperty(required = true, description = "The name of the admin server")
	private String adminServerName;

	@ConfigurationItemProperty(required = true, label = "Host", description = "The host that runs the admin server")
	private Host activeHost;

	public HostSession connectToAdminHost() {
		return HostSessionFactory.getHostSession(activeHost);
	}

	public int executeWlstJythonScript(CommandExecutionCallbackHandler handler, HostSession s, String scriptPath, String... args) {
		final String filesep = activeHost.getOperatingSystemFamily().getFileSeparator();
		final String wlstFilePath = wlJavaHome + filesep + "bin" + filesep + "java";
		final String weblogicJarFilePath = wlHome + filesep + "server" + filesep + "lib" + filesep + "weblogic.jar";

		List<String> cmd = new ArrayList<String>();
		cmd.add(wlstFilePath);
		cmd.add("-classpath");
		cmd.add(weblogicJarFilePath);
		cmd.add("weblogic.WLST");
		cmd.add("-i");
		cmd.add(scriptPath);
		for (String arg : args) {
			cmd.add(arg);
		}
		String[] cmdArray = (String[]) cmd.toArray(new String[cmd.size()]);

		return s.execute(handler, cmdArray);
	}

	public String getConnectUrl() {
		return activeHost.getAddress() + ":" + port;
	}

	public Host getActiveHost() {
		return activeHost;
	}

	public void setActiveHost(Host activeHost) {
		this.activeHost = activeHost;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getUsername() {
		return username;
	}

	public void setUsername(String username) {
		this.username = username;
	}

	public String getPassword() {
		return password;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	public int getPort() {
		return port;
	}

	public void setPort(int port) {
		this.port = port;
	}

	public String getWlHome() {
		return wlHome;
	}

	public void setWlHome(String wlHome) {
		this.wlHome = wlHome;
	}

	public String getWlJavaHome() {
		return wlJavaHome;
	}

	public void setWlJavaHome(String wlJavaHome) {
		this.wlJavaHome = wlJavaHome;
	}

	public String getDomainHome() {
		return StringUtils.defaultIfEmpty(domainHome, getDefaultDomainHome());
	}

	private String getDefaultDomainHome() {
		final String fileSeparator = activeHost.getOperatingSystemFamily().getFileSeparator();
		return wlHome.substring(0, wlHome.lastIndexOf(fileSeparator)) + fileSeparator + "user_projects" + fileSeparator + "domains" + fileSeparator + name;
	}

	public void setDomainHome(String domainHome) {
		this.domainHome = domainHome;
	}

	public String getAdminServerName() {
		return adminServerName;
	}

	public void setAdminServerName(String adminServerName) {
		this.adminServerName = adminServerName;
	}
	
}
