/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wls.ci;

import com.xebialabs.deployit.*;
import com.xebialabs.deployit.ConfigurationItemProperty.Size;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.plugin.wls.step.RetrieveWlsServerStep;
import com.xebialabs.deployit.plugin.wls.step.WlsDomainCheckReachableStep;
import com.xebialabs.deployit.step.HostConnectionStep;
import org.apache.commons.beanutils.BeanComparator;

import java.util.*;

/**
 * A WLS server (Managed or admin) in a {@link WlsDomain WLS domain}.
 * 
 * @see WlsCluster
 * @see WlsDomain
 */
@SuppressWarnings("serial")
@ConfigurationItem(description = "A standard Weblogic Server", discoverable = true)
public class WlsServer extends WlsTarget implements Discoverable<WlsServer> {

	@ConfigurationItemProperty(required = true, description = "Name of the WebLogic Server", identifying = true)
	private String name;

	@ConfigurationItemProperty(required = true, description = "Port for the WebLogic Server")
	private int port;

	@ConfigurationItemProperty(description = "Classpath entries for this server.")
	private String classpath;

	@ConfigurationItemProperty(description = "The arguments for this server, including initial heapsize (e.g. -Xms64m), maxheap size (-Xms256m), and bootclasspath (-Xbootclasspath/p:/var/lib/addons.jar)", size = Size.LARGE)
	private String arguments;

	@ConfigurationItemProperty(description = "Absolute path of log file. Example; /opt/bea/user_projects/domain/managedserver1/ms1.log")
	private String logFileLocation;

	@ConfigurationItemProperty(description = "Enable JVM StdOut to Server Log file")
	private boolean enableJVMLogRedirection = false;

	@ConfigurationItemProperty(required = true, asContainment = true, description = "WebLogic Domain to which this server belongs")
	private WlsDomain domain;

	@ConfigurationItemProperty(required = true, description = "Host on which this server is running")
	private Host host;

	@ConfigurationItemProperty(required = false, label = "Managed server start command", description = "Command that should be executed to start the managed server.", size = Size.LARGE, discoveryParam = true)
	private String startCommand;

	@ConfigurationItemProperty(required = false, label = "Managed server stop command", description = "Command that should be executed to stop the managed server.", size = Size.LARGE, discoveryParam = true)
	private String stopCommand;

	@ConfigurationItemProperty(required = false, description = "Deployment staging mode (default is stage)")
	private WlsStageMode stageMode = WlsStageMode.Stage;

	@SuppressWarnings("unchecked")
	public static List<WlsServer> toNameSortedList(Set<WlsServer> servers) {
		List<WlsServer> nameSortedServers = new ArrayList<WlsServer>(servers);
		Collections.sort(nameSortedServers, new BeanComparator("name"));
		return nameSortedServers;
	}

	public String getTargetObjectName() {
		return "com.bea:Name=" + getName() + ",Type=Server";
	}

	public WlsDomain getDomain() {
		return domain;
	}

	public void setDomain(WlsDomain domain) {
		this.domain = domain;
	}

	public Host getHost() {
		return host;
	}

	public void setHost(Host host) {
		this.host = host;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public int getPort() {
		return port;
	}

	public void setPort(int port) {
		this.port = port;
	}

	public String getClasspath() {
		return classpath;
	}

	public void setClasspath(String classpath) {
		this.classpath = classpath;
	}

	public String getLogFileLocation() {
		return logFileLocation;
	}

	public void setLogFileLocation(String logFileLocation) {
		this.logFileLocation = logFileLocation;
	}

	public boolean isEnableJVMLogRedirection() {
		return enableJVMLogRedirection;
	}

	public void setEnableJVMLogRedirection(boolean enableJVMLogRedirection) {
		this.enableJVMLogRedirection = enableJVMLogRedirection;
	}

	public String getArguments() {
		return arguments;
	}

	public void setArguments(String arguments) {
		this.arguments = arguments;
	}

	private List<Step> getCheckSteps() {
		List<Step> steps = new ArrayList<Step>();
		steps.add(new HostConnectionStep(host));
		steps.add(new WlsDomainCheckReachableStep(this.getDomain()));
		steps.add(new RetrieveWlsServerStep(this));
		// steps.add(new WlsServerCheckReachableStep(this));

		return steps;
	}

	public WlsServer discover(Map<String, Object> info, ChangePlan cp) {
		cp.addSteps(getCheckSteps());
		cp.execute(info);
		return this;
	}

	public WlsServer discoverChildrenInfo(Map<String, Object> info, ChangePlan cp) {
		// children are discovered from the WlsDomain, not here
		return null;
	}

	public void setStageMode(WlsStageMode stageMode) {
		this.stageMode = stageMode;
	}

	public WlsStageMode getStageMode() {
		return (stageMode == null ? WlsStageMode.Stage : stageMode);
	}

	public String getStartCommand() {
		return startCommand;
	}

	public void setStartCommand(String startCommand) {
		this.startCommand = startCommand;
	}

	public String getStopCommand() {
		return stopCommand;
	}

	public void setStopCommand(String stopCommand) {
		this.stopCommand = stopCommand;
	}

}
