/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wls.mapper;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.ci.artifact.NamedDeployableArtifact;
import com.xebialabs.deployit.ci.artifact.mapping.PlaceholderFormat;
import com.xebialabs.deployit.ci.mapping.KeyValuePair;
import com.xebialabs.deployit.plugin.wls.ci.*;
import com.xebialabs.deployit.plugin.wls.step.*;
import com.xebialabs.deployit.steps.CopyStep;
import org.apache.commons.lang.StringUtils;

import java.lang.reflect.Field;
import java.util.*;

/**
 * Helper class handling the generation of Wls Deployment Steps using the stage/no stage mode and deployment plan.
 * Manage Deployment, Undeployment, StopServer.
 */
public class WlsArtifactDeployer {

	public List<? extends Step> generateDeploySteps(List<WlsArtifactMapping> mappings) {
		List<Step> stageSteps = Lists.newArrayList();

		List<Step> noStageSteps = Lists.newArrayList();
		Set<CopyStep> copySteps = Sets.newHashSet();

		for (WlsArtifactMapping mapping : mappings) {
			if (!isNoStageMode(mapping)) {
				//stageMode
				stageSteps.addAll(generateStageModeDeploysSteps(mapping));
			} else {
				//noStageMode
				noStageSteps.addAll(generateNoStageModeDeploysSteps(mapping, copySteps));
			}
		}

		//Merge all steps;
		List<Step> allSteps = new ArrayList<Step>();
		allSteps.addAll(stageSteps);
		allSteps.addAll(copySteps);
		allSteps.addAll(noStageSteps);

		return allSteps;
	}

	private List<? extends Step> generateNoStageModeDeploysSteps(WlsArtifactMapping m, Set<CopyStep> copySteps) {
		final List<Step> steps = Lists.newArrayList();
		final Map<String, String> propertyMap = KeyValuePair.toMap(m.getKeyValuePairs());
		final PlaceholderFormat placeholderFormat = m.getPlaceholderFormat();
		final WlsTarget target = m.getTarget();


		String remoteDestination = "";
		String remoteDeploymentPlanDestination = "";

		for (Host remoteHost : findHosts(target)) {
			String separator = remoteHost.getFileSeparator();
			if (StringUtils.isBlank(m.getStagingDirectory()))
				throw new IllegalArgumentException("With noStage Mode, a staging directory must be filled.");

			remoteDestination = m.getStagingDirectory() + separator + m.getSource().getName() + "." + getExtension(m.getSource());
			final CopyStep copyStepArtifact = new CopyStep(Host.getLocalHost(), m.getSource().getLocation(), remoteHost, remoteDestination, propertyMap, placeholderFormat);
			copyStepArtifact.setCreateToplevelDirectory(true);
			copySteps.add(copyStepArtifact);

			if (m.getDeploymentPlan() != null) {
				if (StringUtils.isBlank(m.getDeploymentPlanStagingDirectory()))
					throw new IllegalArgumentException("With a deployment plan, a deployment plan staging directory must be filled.");
				remoteDeploymentPlanDestination = m.getDeploymentPlanStagingDirectory() + separator + m.getSource().getName() + ".xml";
				final CopyStep copyStepPlan = new CopyStep(Host.getLocalHost(), m.getDeploymentPlan().getLocation(), remoteHost, remoteDeploymentPlanDestination, propertyMap, placeholderFormat);
				copyStepPlan.setCreateToplevelDirectory(true);
				copySteps.add(copyStepPlan);
			}
		}


		if (target.isRunningVersion8()) {
			steps.add(new Wls8DeployNoStageArtifactStep(target, m.getSource(), remoteDestination));
		} else {
			WlsDeployNoStageArtifactStep step = new WlsDeployNoStageArtifactStep(target, m.getSource(), remoteDestination);
			if (m.getDeploymentPlan() != null) {
				step.setRemoteDeploymentPlanPath(remoteDeploymentPlanDestination);
			}
			steps.add(step);
		}

		return steps;
	}

	private List<? extends Step> generateStageModeDeploysSteps(WlsArtifactMapping m) {
		final List<Step> steps = Lists.newArrayList();
		final Map<String, String> propertyMap = KeyValuePair.toMap(m.getKeyValuePairs());
		final PlaceholderFormat placeholderFormat = m.getPlaceholderFormat();
		final WlsTarget target = m.getTarget();


		if (target.isRunningVersion8()) {
			steps.add(new Wls8DeployArtifactStep(target, m.getSource(), propertyMap, placeholderFormat));
		} else {
			WlsDeployArtifactStep step = new WlsDeployArtifactStep(target, m.getSource(), propertyMap, placeholderFormat);
			if (m.getDeploymentPlan() != null) {
				Host activeHost = target.getDomain().getActiveHost();
				String remoteDeploymentPlanDestination = m.getDeploymentPlanStagingDirectory() + activeHost + m.getSource().getName() + ".xml";
				if (StringUtils.isBlank(m.getDeploymentPlanStagingDirectory()))
					throw new IllegalArgumentException("With a deployment plan, a deployment plan staging directory must be filled.");

				step.setDeploymentPlan(m.getDeploymentPlan(), remoteDeploymentPlanDestination);
			}
			steps.add(step);
		}
		return steps;
	}

	private String getExtension(NamedDeployableArtifact artifact) {
		try {
			final Field field = artifact.getClass().getField("ARCHIVE_EXTENSION");
			return "." + field.get(artifact).toString();
		} catch (Exception e) {
			throw new RuntimeException("Unexpected artifact " + artifact, e);
		}
	}

	private Set<Host> findHosts(WlsTarget target) {

		Set<Host> hosts = new HashSet<Host>();

		if (target instanceof WlsServer) {
			WlsServer server = (WlsServer) target;
			hosts.add(server.getHost());
		}

		if (target instanceof WlsCluster) {
			WlsCluster cluster = (WlsCluster) target;
			for (WlsServer server : cluster.getServers())
				hosts.add(server.getHost());
		}

		return hosts;
	}


	private boolean isNoStageMode(WlsArtifactMapping artefactMapping) {
		final WlsTarget target = artefactMapping.getTarget();
		if (target instanceof WlsCluster) {
			return artefactMapping.getStageMode().equals(WlsStageMode.NoStage);
		}
		if (target instanceof WlsServer) {
			WlsServer server = (WlsServer) target;
			return server.getStageMode().equals(WlsStageMode.NoStage) || artefactMapping.getStageMode().equals(WlsStageMode.NoStage);
		}
		return artefactMapping.getStageMode().equals(WlsStageMode.NoStage);
	}

	public List<? extends Step> generateUndeploysSteps(List<? extends WlsArtifactMapping> mappings) {
		List<Step> steps = Lists.newArrayList();
		for (WlsArtifactMapping mapping : mappings)
			steps.addAll(generateUndeploysSteps(mapping));

		return steps;
	}

	public List<? extends Step> generateStartSteps(List<? extends WlsArtifactMapping> mappings) {
		List<Step> steps = Lists.newArrayList();
		for (WlsArtifactMapping mapping : mappings)
			steps.addAll(generateStartSteps(mapping));
		return steps;
	}

	private List<? extends Step> generateUndeploysSteps(WlsArtifactMapping mapping) {
		final List<Step> steps = Lists.newArrayList();
		final WlsTarget target = mapping.getTarget();
		final NamedDeployableArtifact artifact = mapping.getSource();
		switch (mapping.getDeploymentStrategy()) {
			case CLASSIC:
				if (target.isRunningVersion8()) {
					steps.add(new Wls8UndeployArtifactStep(target, artifact));
				} else {
					steps.add(new StopApplicationStep(target, artifact));
					steps.add(new WlsUndeployArtifactStep(target, artifact));
				}
				break;
			case STOP_START:
				if (target.isRunningVersion8()) {
					steps.add(new Wls8UndeployArtifactStep(target, artifact));
				} else {
					steps.add(new WlsUndeployArtifactStep(target, artifact));
				}
				break;
			case SIDE_BY_SIDE:
				throw new RuntimeException("the side by side strategy is not supported yet ");
		}
		return steps;
	}


	private List<? extends Step> generateStartSteps(WlsArtifactMapping mapping) {
		List<Step> steps = Lists.newArrayList();
		final WlsTarget target = mapping.getTarget();
		switch (mapping.getDeploymentStrategy()) {
			case CLASSIC:
				if (target.isRunningVersionGreaterThan9())
					steps.add(new StartApplicationStep(target, mapping.getSource()));
				break;
			case STOP_START:
				break;
			case SIDE_BY_SIDE:
				throw new RuntimeException("the side by side strategy is not supported yet ");
		}
		return steps;
	}
}

