package com.xebialabs.deployit.core.upgrade.configuration

import com.xebialabs.deployit.core.upgrade.configuration.common.BaseConfigurationUpgrader
import com.xebialabs.deployit.server.api.upgrade.Version
import org.yaml.snakeyaml.{DumperOptions, Yaml}

import java.io.{File, FileInputStream}
import java.util
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

class ArtifactResolver2430ArchiveExtensionUpgrader extends BaseConfigurationUpgrader {

  override def destinationFileName: String = "deploy-artifact-resolver.yaml"

  override def configurationFileDescription: String = "Artifact Resolver Configuration"

  val XL_ARCHIVE_EXTENSIONS_PATH = "archive-extensions"

  override def upgradeVersion(): Version = Version
    .valueOf("central-config", "24.3.5")

  override def doUpgrade(): Boolean = Try {
    updateConfiguration()
  } match {
    case Success(_) =>
      logger.info(s"Updating archive-extensions ' from $destinationFileName")
      true
    case Failure(exception: Exception) =>
      logger.info(s"$destinationFileName configuration cannot be updated: " + exception.getMessage)
      false
    case _ =>
      false
  }

  override protected def updateConfiguration(): Unit = {
    val configFile = new File(s"$centralConfFolder/$destinationFileName")
    var originalMap = readYamlFile(configFile)
    val result = findKeyInMap(originalMap, XL_ARCHIVE_EXTENSIONS_PATH)
    var updatedMap: Map[String, String] = result match {
      case Some(value: util.Map[_, _]) => value.asInstanceOf[util.Map[String, String]].asScala.toMap
      case _ => Map.empty[String, String]
    }
    if(updatedMap.nonEmpty){
      if(updatedMap.contains("tar.gz")){
        updatedMap = updatedMap + ("tar-gz" -> updatedMap("tar.gz")) - "tar.gz"
      }
      if(updatedMap.contains("tar.bz2")){
        updatedMap = updatedMap + ("tar-bz2" -> updatedMap("tar.bz2")) - "tar.bz2"
      }
      updateKeyInMap(originalMap, XL_ARCHIVE_EXTENSIONS_PATH, updatedMap.asJava)
      // Serialize the updated map back to YAML
      val dumperOptions = new DumperOptions()
      dumperOptions.setPrettyFlow(true)
      dumperOptions.setDefaultFlowStyle(DumperOptions.FlowStyle.BLOCK)
      val yaml = new Yaml(dumperOptions)

      val writer = new java.io.FileWriter(configFile)
      try {
        yaml.dump(originalMap, writer)
      } finally {
        writer.close()
      }
    }
  }

  private def readYamlFile(file: File): util.Map[String, AnyRef] = {
    val dumperOptions = new DumperOptions()
    dumperOptions.setPrettyFlow(true)
    dumperOptions.setDefaultFlowStyle(DumperOptions.FlowStyle.BLOCK)
    val parser = new Yaml(dumperOptions)
    parser.load(new FileInputStream(file))
  }

  def findKeyInMap(map: util.Map[String, AnyRef], targetKey: String): Option[util.Map[String, AnyRef]] = {
    def search(map: util.Map[String, AnyRef]): Option[util.Map[String, AnyRef]] = {
      map.asScala.foreach {
        case (key, value) if key == targetKey => return Some(value.asInstanceOf[util.Map[String, AnyRef]])
        case (_, value: util.Map[_, _]) =>
          val result = search(value.asInstanceOf[util.Map[String, AnyRef]])
          if (result.isDefined) return result
        case (_, value: util.List[_]) =>
          value.asScala.foreach {
            case nestedMap: util.Map[_, _] =>
              val result = search(nestedMap.asInstanceOf[util.Map[String, AnyRef]])
              if (result.isDefined) return result
            case _ => // Do nothing
          }
        case _ => // Do nothing
      }
      None
    }

    search(map)
  }

  def updateKeyInMap(map: util.Map[String, AnyRef], targetKey: String, newValue: AnyRef): Boolean = {
    def searchAndUpdate(map: util.Map[String, AnyRef]): Boolean = {
      map.asScala.foreach {
        case (key, value) if key == targetKey =>
          map.put(key, newValue)
          return true
        case (_, value: util.Map[_, _]) =>
          if (searchAndUpdate(value.asInstanceOf[util.Map[String, AnyRef]])) return true
        case (_, value: util.List[_]) =>
          value.asScala.foreach {
            case nestedMap: util.Map[_, _] =>
              if (searchAndUpdate(nestedMap.asInstanceOf[util.Map[String, AnyRef]])) return true
            case _ => // Do nothing
          }
        case _ => // Do nothing
      }
      false
    }

    searchAndUpdate(map)
  }
}
