package com.xebialabs.deployit.core.upgrade.configuration

import com.xebialabs.deployit.core.upgrade.configuration.common.BaseConfigurationUpgrader
import com.xebialabs.deployit.server.api.upgrade.Version
import org.yaml.snakeyaml.{DumperOptions, Yaml}

import java.io.{File, FileInputStream}
import java.util
import scala.util.{Failure, Success, Try}
import com.xebialabs.deployit.core.upgrade.configuration.CacheProperties2410._

class CachesProperties2410Upgrader extends BaseConfigurationUpgrader {

  override def destinationFileName: String = "deploy-caches.yaml"

  override def configurationFileDescription: String = "Caches config"

  override def upgradeVersion(): Version = Version
    .valueOf("central-config", "24.1.0")

  override def doUpgrade(): Boolean = Try {
    updateConfiguration()
  } match {
    case Success(_) =>
      logger.info(s"Updating deploy caches configuration' from $destinationFileName")
      true
    case Failure(exception: Exception) =>
      logger.info(s"$destinationFileName configuration cannot be updated: " + exception.getMessage)
      false
    case _ =>
      false
  }

  override protected def updateConfiguration(): Unit = {

    val configFile = new File(s"$centralConfFolder/$destinationFileName")
    val updatedProperties: Map[String, AnyRef] = getUpdateProperties(configFile)
    updateCentralConfigurationProperties(updatedProperties)
    deleteCentralConfigurationProperties(
      Seq(
        DEPLOY_CACHE_ENABLED,
        DEPLOY_CACHE_PROVIDER_CONFIG,
        DEPLOY_CACHE_PROVIDER_CONFIG_FILE,
        DEPLOY_CACHE_CIRCUIT_BREAK_CONFIG_FAILURE_RATE,
        DEPLOY_CACHE_CIRCUIT_BREAK_CONFIG_WAIT,
        DEPLOY_CACHE_CIRCUIT_BREAK_CONFIG_MINI,
        DEPLOY_CACHE_CIRCUIT_BREAK_CONFIG_SLIDING,
        DEPLOY_CACHE_CIRCUIT_BREAK_CONFIG_PERMITTED,
        DEPLOY_CACHE_LICENSE_CACHE_ENABLED,
        DEPLOY_CACHE_LICENSE_CACHE_MGT,
        DEPLOY_CACHE_LICENSE_CACHE_STATS,
        DEPLOY_CACHE_LICENSE_CACHE_TTL,
        DEPLOY_CACHE_CI_PK_ENABLED,
        DEPLOY_CACHE_CI_PK_MGT,
        DEPLOY_CACHE_CI_PK_STATS,
        DEPLOY_CACHE_CI_PK_TTL,
        DEPLOY_CACHE_CI_PATH_ENABLED,
        DEPLOY_CACHE_CI_PATH_MGT,
        DEPLOY_CACHE_CI_PATH_STATS,
        DEPLOY_CACHE_CI_PATH_TTL,
        DEPLOY_CACHE_CI_PROP_ENABLED,
        DEPLOY_CACHE_CI_PROP_MGT,
        DEPLOY_CACHE_CI_PROP_STATS,
        DEPLOY_CACHE_CI_PROP_TTL
      )
    )
  }

  private def getUpdateProperties(configFile: File): Map[String, AnyRef] = {
    val properties: Map[String, List[String]] = Map(
      "deploy.caches.app-caches.enabled" -> DEPLOY_CACHE_ENABLED.split("\\.").toList,
      "deploy.caches.app-caches.provider-configuration.provider" -> DEPLOY_CACHE_PROVIDER_CONFIG.split("\\.").toList,
      "deploy.caches.app-caches.provider-configuration.provider-configuration-file" -> DEPLOY_CACHE_PROVIDER_CONFIG_FILE.split("\\.").toList,
      "deploy.caches.app-caches.circuit-breaker-configuration.failure-rate-threshold-percentage" -> DEPLOY_CACHE_CIRCUIT_BREAK_CONFIG_FAILURE_RATE.split("\\.").toList,
      "deploy.caches.app-caches.circuit-breaker-configuration.wait-duration-in-open-state-seconds" -> DEPLOY_CACHE_CIRCUIT_BREAK_CONFIG_WAIT.split("\\.").toList,
      "deploy.caches.app-caches.circuit-breaker-configuration.minimum-number-of-calls" -> DEPLOY_CACHE_CIRCUIT_BREAK_CONFIG_MINI.split("\\.").toList,
      "deploy.caches.app-caches.circuit-breaker-configuration.sliding-window-size" -> DEPLOY_CACHE_CIRCUIT_BREAK_CONFIG_SLIDING.split("\\.").toList,
      "deploy.caches.app-caches.circuit-breaker-configuration.permitted-number-of-calls-in-half-open-state" -> DEPLOY_CACHE_CIRCUIT_BREAK_CONFIG_PERMITTED.split("\\.").toList,
      "deploy.caches.app-caches.license-cache-configuration.enabled" -> DEPLOY_CACHE_LICENSE_CACHE_ENABLED.split("\\.").toList,
      "deploy.caches.app-caches.license-cache-configuration.management-enabled" -> DEPLOY_CACHE_LICENSE_CACHE_MGT.split("\\.").toList,
      "deploy.caches.app-caches.license-cache-configuration.statistics-enabled" -> DEPLOY_CACHE_LICENSE_CACHE_STATS.split("\\.").toList ,
      "deploy.caches.app-caches.license-cache-configuration.ttl-minutes" -> DEPLOY_CACHE_LICENSE_CACHE_TTL.split("\\.").toList,
      "deploy.caches.app-caches.ci-pk-cache-configuration.enabled" -> DEPLOY_CACHE_CI_PK_ENABLED.split("\\.").toList,
      "deploy.caches.app-caches.ci-pk-cache-configuration.management-enabled" -> DEPLOY_CACHE_CI_PK_MGT.split("\\.").toList,
      "deploy.caches.app-caches.ci-pk-cache-configuration.statistics-enabled" -> DEPLOY_CACHE_CI_PK_STATS.split("\\.").toList,
      "deploy.caches.app-caches.ci-pk-cache-configuration.ttl-minutes" -> DEPLOY_CACHE_CI_PK_TTL.split("\\.").toList,
      "deploy.caches.app-caches.ci-path-cache-configuration.enabled" -> DEPLOY_CACHE_CI_PATH_ENABLED.split("\\.").toList,
      "deploy.caches.app-caches.ci-path-cache-configuration.management-enabled" -> DEPLOY_CACHE_CI_PATH_MGT.split("\\.").toList,
      "deploy.caches.app-caches.ci-path-cache-configuration.statistics-enabled" -> DEPLOY_CACHE_CI_PATH_STATS.split("\\.").toList,
      "deploy.caches.app-caches.ci-path-cache-configuration.ttl-minutes" -> DEPLOY_CACHE_CI_PATH_TTL.split("\\.").toList,
      "deploy.caches.app-caches.ci-properties-cache-configuration.enabled" -> DEPLOY_CACHE_CI_PROP_ENABLED.split("\\.").toList,
      "deploy.caches.app-caches.ci-properties-cache-configuration.management-enabled" -> DEPLOY_CACHE_CI_PROP_MGT.split("\\.").toList,
      "deploy.caches.app-caches.ci-properties-cache-configuration.statistics-enabled" -> DEPLOY_CACHE_CI_PROP_STATS.split("\\.").toList,
      "deploy.caches.app-caches.ci-properties-cache-configuration.ttl-minutes" -> DEPLOY_CACHE_CI_PROP_TTL.split("\\.").toList,
    )
    var updatedProperties = Map[String, AnyRef]()
    val originalMap = readYamlFile(configFile)
    val map = if (originalMap.containsKey("deploy.caches"))
      new util.LinkedHashMap[String, AnyRef]() {
        put("deploy", new util.LinkedHashMap[String, AnyRef]() {
          put("caches", originalMap.getOrDefault("deploy.caches", new util.LinkedHashMap[String, AnyRef]()))
        })
      }
    else originalMap
    for ((key, values) <- properties) {
      def hasKey: Boolean = try {
        getValue
        true
      } catch {
        case _: Exception => false
      }

      def getValue: AnyRef = {
        val v = values.dropRight(1).foldLeft(map) { (acc, values) =>
          acc.get(values).asInstanceOf[util.Map[String, AnyRef]]
        }
        v.get(values.lastOption.orNull)
      }

      if (hasKey) {
        updatedProperties += (key -> getValue)
      }
    }
    updatedProperties
  }

  private def readYamlFile(file: File): util.Map[String, AnyRef] = {
    val dumperOptions = new DumperOptions()
    dumperOptions.setPrettyFlow(true)
    dumperOptions.setDefaultFlowStyle(DumperOptions.FlowStyle.BLOCK)
    val parser = new Yaml(dumperOptions)
    parser.load(new FileInputStream(file))
  }

}

object CacheProperties2410 {
  val DEPLOY_CACHE_ENABLED = "deploy.caches.enabled"
  val DEPLOY_CACHE_PROVIDER_CONFIG = "deploy.caches.provider-configuration.provider"
  val DEPLOY_CACHE_PROVIDER_CONFIG_FILE = "deploy.caches.provider-configuration.provider-configuration-file"
  val DEPLOY_CACHE_CIRCUIT_BREAK_CONFIG_FAILURE_RATE = "deploy.caches.circuit-breaker-configuration.failure-rate-threshold-percentage"
  val DEPLOY_CACHE_CIRCUIT_BREAK_CONFIG_WAIT = "deploy.caches.circuit-breaker-configuration.wait-duration-in-open-state-seconds"
  val DEPLOY_CACHE_CIRCUIT_BREAK_CONFIG_MINI = "deploy.caches.circuit-breaker-configuration.minimum-number-of-calls"
  val DEPLOY_CACHE_CIRCUIT_BREAK_CONFIG_SLIDING = "deploy.caches.circuit-breaker-configuration.sliding-window-size"
  val DEPLOY_CACHE_CIRCUIT_BREAK_CONFIG_PERMITTED = "deploy.caches.circuit-breaker-configuration.permitted-number-of-calls-in-half-open-state"
  val DEPLOY_CACHE_LICENSE_CACHE_ENABLED = "deploy.caches.license-cache-configuration.enabled"
  val DEPLOY_CACHE_LICENSE_CACHE_MGT = "deploy.caches.license-cache-configuration.management-enabled"
  val DEPLOY_CACHE_LICENSE_CACHE_STATS = "deploy.caches.license-cache-configuration.statistics-enabled"
  val DEPLOY_CACHE_LICENSE_CACHE_TTL = "deploy.caches.license-cache-configuration.ttl-minutes"
  val DEPLOY_CACHE_CI_PK_ENABLED = "deploy.caches.ci-pk-cache-configuration.enabled"
  val DEPLOY_CACHE_CI_PK_MGT = "deploy.caches.ci-pk-cache-configuration.management-enabled"
  val DEPLOY_CACHE_CI_PK_STATS = "deploy.caches.ci-pk-cache-configuration.statistics-enabled"
  val DEPLOY_CACHE_CI_PK_TTL = "deploy.caches.ci-pk-cache-configuration.ttl-minutes"
  val DEPLOY_CACHE_CI_PATH_ENABLED = "deploy.caches.ci-path-cache-configuration.enabled"
  val DEPLOY_CACHE_CI_PATH_MGT = "deploy.caches.ci-path-cache-configuration.management-enabled"
  val DEPLOY_CACHE_CI_PATH_STATS = "deploy.caches.ci-path-cache-configuration.statistics-enabled"
  val DEPLOY_CACHE_CI_PATH_TTL = "deploy.caches.ci-path-cache-configuration.ttl-minutes"
  val DEPLOY_CACHE_CI_PROP_ENABLED = "deploy.caches.ci-properties-cache-configuration.enabled"
  val DEPLOY_CACHE_CI_PROP_MGT = "deploy.caches.ci-properties-cache-configuration.management-enabled"
  val DEPLOY_CACHE_CI_PROP_STATS = "deploy.caches.ci-properties-cache-configuration.statistics-enabled"
  val DEPLOY_CACHE_CI_PROP_TTL ="deploy.caches.ci-properties-cache-configuration.ttl-minutes"
}
