package com.fasterxml.jackson.databind.deser.std;

import java.io.IOException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.core.JsonTokenId;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.cfg.CoercionAction;
import com.fasterxml.jackson.databind.jsontype.TypeDeserializer;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.type.LogicalType;

/**
 * Code used from NumberDeserializers.BooleanDeserializer
 */
public final class BooleanDeserializerWithYesNoSupport extends NumberDeserializers.PrimitiveOrWrapperDeserializer<Boolean> {
    private static final long serialVersionUID = 1L;

    public final static BooleanDeserializerWithYesNoSupport primitiveInstance = new BooleanDeserializerWithYesNoSupport(Boolean.TYPE, Boolean.FALSE);
    public final static BooleanDeserializerWithYesNoSupport wrapperInstance = new BooleanDeserializerWithYesNoSupport(Boolean.class, null);

    public static SimpleModule createModule() {
        SimpleModule booleanIgnoreCaseModule = new SimpleModule();
        booleanIgnoreCaseModule.addDeserializer(Boolean.class, BooleanDeserializerWithYesNoSupport.wrapperInstance);
        booleanIgnoreCaseModule.addDeserializer(boolean.class, BooleanDeserializerWithYesNoSupport.primitiveInstance);
        return booleanIgnoreCaseModule;
    }

    public BooleanDeserializerWithYesNoSupport(Class<Boolean> cls, Boolean nvl) {
        super(cls, LogicalType.OtherScalar, nvl, Boolean.FALSE);
    }

    @Override
    public Boolean deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
        JsonToken t = p.currentToken();
        if (t == JsonToken.VALUE_TRUE) {
            return Boolean.TRUE;
        }
        if (t == JsonToken.VALUE_FALSE) {
            return Boolean.FALSE;
        }
        if (_primitive) {
            return _parseBooleanPrimitiveWithYesNo(p, ctxt);
        }
        return _parseBooleanWithYesNo(p, ctxt, _valueClass);
    }

    // Since we can never have type info ("natural type"; String, Boolean, Integer, Double):
    // (is it an error to even call this version?)
    @Override
    public Boolean deserializeWithType(JsonParser p, DeserializationContext ctxt,
                                       TypeDeserializer typeDeserializer)
            throws IOException {
        JsonToken t = p.currentToken();
        if (t == JsonToken.VALUE_TRUE) {
            return Boolean.TRUE;
        }
        if (t == JsonToken.VALUE_FALSE) {
            return Boolean.FALSE;
        }
        if (_primitive) {
            return _parseBooleanPrimitiveWithYesNo(p, ctxt);
        }
        return _parseBooleanWithYesNo(p, ctxt, _valueClass);
    }

    protected final boolean _parseBooleanPrimitiveWithYesNo(JsonParser p, DeserializationContext ctxt)
            throws IOException {
        String text;
        switch (p.currentTokenId()) {
            case JsonTokenId.ID_STRING:
                text = p.getText();
                break;
            case JsonTokenId.ID_NUMBER_INT:
                // may accept ints too, (0 == false, otherwise true)

                // call returns `null`, Boolean.TRUE or Boolean.FALSE so:
                return Boolean.TRUE.equals(_coerceBooleanFromInt(p, ctxt, Boolean.TYPE));
            case JsonTokenId.ID_TRUE: // usually caller should have handled but:
                return true;
            case JsonTokenId.ID_FALSE:
                return false;
            case JsonTokenId.ID_NULL: // null fine for non-primitive
                _verifyNullForPrimitive(ctxt);
                return false;
            // 29-Jun-2020, tatu: New! "Scalar from Object" (mostly for XML)
            case JsonTokenId.ID_START_OBJECT:
                text = ctxt.extractScalarFromObject(p, this, Boolean.TYPE);
                break;
            case JsonTokenId.ID_START_ARRAY:
                // 12-Jun-2020, tatu: For some reason calling `_deserializeFromArray()` won't work so:
                if (ctxt.isEnabled(DeserializationFeature.UNWRAP_SINGLE_VALUE_ARRAYS)) {
                    p.nextToken();
                    final boolean parsed = _parseBooleanPrimitiveWithYesNo(p, ctxt);
                    _verifyEndArrayForSingle(p, ctxt);
                    return parsed;
                }
                // fall through
            default:
                return ((Boolean) ctxt.handleUnexpectedToken(Boolean.TYPE, p)).booleanValue();
        }

        final CoercionAction act = _checkFromStringCoercion(ctxt, text,
                LogicalType.Boolean, Boolean.TYPE);
        if (act == CoercionAction.AsNull) {
            _verifyNullForPrimitive(ctxt);
            return false;
        }
        if (act == CoercionAction.AsEmpty) {
            return false;
        }
        text = text.trim();
        final int len = text.length();

        // For [databind#1852] allow some case-insensitive matches (namely,
        // true/True/TRUE, false/False/FALSE
        if (len == 4) {
            if (_isTrue(text)) {
                return true;
            }
        } else if (len == 5) {
            if (_isFalse(text)) {
                return false;
            }
        } else if (len == 3) {
            if (_isYes(text)) {
                return true;
            }
        } else if (len == 2) {
            if (_isNo(text)) {
                return false;
            }
        }
        if (_hasTextualNull(text)) {
            _verifyNullForPrimitiveCoercion(ctxt, text);
            return false;
        }
        Boolean b = (Boolean) ctxt.handleWeirdStringValue(Boolean.TYPE, text,
                "only \"true\"/\"True\"/\"TRUE\" or \"false\"/\"False\"/\"FALSE\" recognized");
        return Boolean.TRUE.equals(b);
    }

    protected final Boolean _parseBooleanWithYesNo(JsonParser p, DeserializationContext ctxt,
                                                   Class<?> targetType)
            throws IOException {
        String text;
        switch (p.currentTokenId()) {
            case JsonTokenId.ID_STRING:
                text = p.getText();
                break;
            case JsonTokenId.ID_NUMBER_INT:
                // may accept ints too, (0 == false, otherwise true)
                return _coerceBooleanFromInt(p, ctxt, targetType);
            case JsonTokenId.ID_TRUE:
                return true;
            case JsonTokenId.ID_FALSE:
                return false;
            case JsonTokenId.ID_NULL: // null fine for non-primitive
                return null;
            // 29-Jun-2020, tatu: New! "Scalar from Object" (mostly for XML)
            case JsonTokenId.ID_START_OBJECT:
                text = ctxt.extractScalarFromObject(p, this, targetType);
                break;
            case JsonTokenId.ID_START_ARRAY: // unwrapping / from-empty-array coercion?
                return (Boolean) _deserializeFromArray(p, ctxt);
            default:
                return (Boolean) ctxt.handleUnexpectedToken(targetType, p);
        }

        final CoercionAction act = _checkFromStringCoercion(ctxt, text,
                LogicalType.Boolean, targetType);
        if (act == CoercionAction.AsNull) {
            return null;
        }
        if (act == CoercionAction.AsEmpty) {
            return false;
        }
        text = text.trim();
        final int len = text.length();

        // For [databind#1852] allow some case-insensitive matches (namely,
        // true/True/TRUE, false/False/FALSE
        if (len == 4) {
            if (_isTrue(text)) {
                return true;
            }
        } else if (len == 5) {
            if (_isFalse(text)) {
                return false;
            }
        } else if (len == 3) {
            if (_isYes(text)) {
                return true;
            }
        } else if (len == 2) {
            if (_isNo(text)) {
                return false;
            }
        }
        if (_checkTextualNull(ctxt, text)) {
            return null;
        }
        return (Boolean) ctxt.handleWeirdStringValue(targetType, text,
                "only \"true\" or \"false\" or \"yes\" or \"no\" recognized");
    }

    protected boolean _isYes(String text) {
        return "yes".equalsIgnoreCase(text);
    }

    protected boolean _isNo(String text) {
        return "no".equalsIgnoreCase(text);
    }
}
