package com.xebialabs.deployit.core.upgrade.configuration

import com.typesafe.config.Config
import com.xebialabs.deployit.core.upgrade.configuration.common.ConfigUtils.loadXlDeployConf
import com.xebialabs.deployit.core.upgrade.configuration.common.{BaseConfigurationUpgrader, ConfigUtils}
import com.xebialabs.deployit.server.api.upgrade.Version

import java.util
import scala.util.{Failure, Success, Try}

/*
Configuration Upgrader for 10.3 for left over configurations in xl-deploy.conf to be moved to
deploy-artifact-resolver.yaml
*/
class Deploy1040ArtifactPlaceholderArchiveExtensionsUpgrader extends BaseConfigurationUpgrader {

  protected var xlDeployConfOption: Option[Config] = None

  override def destinationFileName: String = "deploy-artifact-resolver.yaml"

  override def configurationFileDescription: String = "Artifact placeholder archive extensions config"

  override def upgradeVersion(): Version = Version.valueOf("central-config", "10.4.0")

  override def doUpgrade(): Boolean = Try {
    updateConfiguration()
  } match {
    case Success(_) =>
      logger.debug(s"$configurationFileDescription configuration has been updated successfully.")
      true
    case Failure(exception: Exception) =>
      logger.debug(s"$configurationFileDescription configuration cannot be updated: " + exception.getMessage)
      false
    case _ =>
      false
  }

  override protected def updateConfiguration(): Unit = {
    xlDeployConfOption = loadXlDeployConf(xlDeployConfFile)
    xlDeployConfOption.map(conf => {
      Try {
        // Read from xl-deploy and create map of configs
        val confToUpdate = getConfToUpdate(conf)
        updateCentralConfigurationProperties(confToUpdate)
        // clean up
        excludeXlDeployConfSections(List(XL_ARCHIVE_EXTENSIONS_PATH))
        logger.debug(s"Moved $confToUpdate from xl-deploy.conf to deploy-artifact.yaml")
      } match {
        case Failure(e) =>
          logger.error(s"Failure in moving configuration form xl-deploy.conf to : $destinationFile", e)
          throw e
        case _ =>
      }
    })
  }

  /**
   * Read from xl-deploy.conf and create map for the specific set of confs to be updated to the destination file
   *
   * @param deployConf - conf from xl-deploy.conf
   * @return map of subset config from server config
   */
  def getConfToUpdate(deployConf: Config): Map[String, AnyRef] = {
    if (deployConf.hasPath(XL_ARCHIVE_EXTENSIONS_PATH)) {

      import scala.jdk.CollectionConverters._

      ConfigUtils.getValueOrDefault(deployConf, XL_ARCHIVE_EXTENSIONS_PATH, new util.HashMap[String, String]())
        .asInstanceOf[util.Map[String, String]]
        .asScala
        .map { case (key, value) =>
          (DEPLOY_ARCHIVE_EXTENSIONS_PATH + "." + key, value)
        }.toMap
    } else {
      Map.empty
    }
  }

  val DEPLOY_ARCHIVE_EXTENSIONS_PATH = "deploy.artifact.placeholders.archive-extensions"
  val XL_ARCHIVE_EXTENSIONS_PATH = "xl.xldeploy.placeholders.archive-extensions"

}
