package com.xebialabs.deployit.support.report

import com.xebialabs.xlplatform.cluster.membership.storage.ClusterMembershipManagement.{Data, Seed}
import com.xebialabs.xlplatform.cluster.{ClusterConfig, XlCluster}
import com.xebialabs.xlplatform.cluster.ClusterMode.{Full, HotStandby, Standalone}
import com.xebialabs.xlplatform.config.ProductConfiguration
import com.xebialabs.xlplatform.support.report.ReportDataProvider
import grizzled.slf4j.Logging

import scala.concurrent.duration._
import scala.jdk.CollectionConverters._
import scala.concurrent.{Await, ExecutionContextExecutor}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util
import scala.beans.BeanProperty
import scala.language.postfixOps

@Component
class ClusterDataProvider @Autowired() (val xldProductConfiguration: ProductConfiguration)
                                        extends ReportDataProvider with Logging {
  val name: String = "cluster"
  override def collectStatistics: Map[String, Any] = {
    val clusterStatistics = new ServerClusterInfo()
    val systemClusterConfig = xldProductConfiguration.configuration.getConfig("deploy.cluster")
    val clusterConfig =  ClusterConfig(systemClusterConfig)

    clusterStatistics.setMode(clusterConfig.mode.toString)
    clusterConfig.mode match {
      case Standalone =>
        clusterStatistics.setNumberOfNodes(1)
      case HotStandby | Full =>
        clusterStatistics.setIsCluster(true)
        val mgmtActorSystem = XlCluster.getActorSystem
        val cluster = XlCluster.getCluster
        val management = XlCluster.getMembershipManagement
        implicit val executor: ExecutionContextExecutor = mgmtActorSystem.dispatcher
        try {
          val releaseActorReceive: FiniteDuration = 60000 millis
          val seeds = Await.result(management.listActiveSeeds(cluster), releaseActorReceive)
          val dbNodes = seeds match {
            case Data(seeds: Seq[Seed]) => seeds.map(_.address.toString)
            case _ => Seq.empty
          }
          clusterStatistics.setSeedNodes(dbNodes.toList.asJava)
          clusterStatistics.setNumberOfNodes(dbNodes.length)
        } catch {
          case e: concurrent.TimeoutException => logger.warn("Timeout encountered querying active seeds, unable to collect all cluster statistics.")
          case _: Throwable => logger.warn("Exception encountered querying active seeds, unable to collect all cluster statistics.")
        }
        clusterStatistics.setActorSystem(mgmtActorSystem.name)
        clusterStatistics.setPekkoNodes(cluster.state.members.map(member => s"${member.address.toString} ${member.status}").toList.asJava)
        clusterStatistics.setLeader(cluster.state.leader.map(_.toString).getOrElse("No leader"))
      case other =>
        throw new IllegalArgumentException(s"Cluster mode $other is not supported.")
    }
    Map[String, Any]("cluster" -> clusterStatistics)
  }
}

class ServerClusterInfo {
  @BeanProperty
  var mode: String = _

  @BeanProperty
  var isCluster: Boolean = false

  @BeanProperty
  var numberOfNodes: Int = 0

  @BeanProperty
  var actorSystem: String = _

  @BeanProperty
  var leader: String = _

  @BeanProperty
  var seedNodes: util.List[String] = new util.ArrayList[String]()

  @BeanProperty
  var pekkoNodes: util.List[String] = new util.ArrayList[String]()
}
