package com.xebialabs.gradle.defaults.tasks

import com.xebialabs.gradle.extensions.XLReleasePluginExtension
import org.gradle.api.Action
import org.gradle.api.DefaultTask
import org.gradle.api.GradleException
import org.gradle.api.Project
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ModuleVersionIdentifier
import org.gradle.api.artifacts.result.ResolvedDependencyResult
import org.gradle.api.artifacts.result.UnresolvedDependencyResult
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.TaskAction

class CheckDependencyVersions extends DefaultTask {
  @Input
  Set<Project> projects

  static final ArrayList<String> ORDERED_PRERELEASES = ['snapshot', 'alpha', 'beta', 'rc', '']

  static def canBeResolved(Configuration configuration) {
    configuration.metaClass.respondsTo(configuration, "isCanBeResolved") ? configuration.isCanBeResolved() : true
  }

  static def getPrerelease(String version) {
    CheckDependencyVersions.ORDERED_PRERELEASES.find { version.toLowerCase().contains(it) }
  }

  // Workaround. Check: https://github.com/gradle/gradle/issues/6854
  static def workaround = { ->
    if (name.startsWith("incrementalScalaAnalysis")) {
      extendsFrom = []
    }
  }

  static def getVersionExclusions(Project project) {
    project.extensions.getByType(XLReleasePluginExtension).excludeFromVersionCheck
  }

  static def getConfigurationExclusions(Project project) {
    project.extensions.getByType(XLReleasePluginExtension).excludeConfigurations
  }

  static def isModuleExcluded(ModuleVersionIdentifier module, Map<String, String> rule) {
    rule.with {
      if (groupId && artifactId) {
        module.group == groupId && module.name == artifactId
      } else if (groupId) {
        module.group == groupId
      } else false
    }
  }

  static def parseRule(Object rule) {
    switch (rule) {
      case String:
        def split = (rule as String).split(":")
        [groupId: split[0], artifactId: split.length > 1 ? split[1] : null]
        break
      case Map: rule as Map<String, String>; break
      default: new HashMap<String, String>()
    }
  }

  static def dependencyExcludedChecker(Project project) {
    def excludedArtifacts = getVersionExclusions(project).plus(getVersionExclusions(project.rootProject))
    def excludedList = new HashSet<String>()
    return [
      excluded: excludedList,
      checker : { ModuleVersionIdentifier module ->
        excludedArtifacts.any { rule ->
          def excluded = isModuleExcluded(module, parseRule(rule))
          if (excluded) {
            excludedList.add("${module.group}:${module.name}:${module.version}")
          }
          excluded
        }
      }
    ]
  }

  static def configurationFilter(Project project) {
    def excludedConfigurations = getConfigurationExclusions(project).plus(getConfigurationExclusions(project.rootProject))
    return { configuration -> !excludedConfigurations.contains(configuration.name) }
  }

  private def handleResolvedDependency(Project currentProject,
                                       ResolvedDependencyResult dependency,
                                       Closure<Boolean> checkDependencyExclusion) {
    def projectVersion = currentProject.version.toString()
    def projectPreleaseComponent = getPrerelease(projectVersion)
    def module = dependency.selected.moduleVersion
    if (!checkDependencyExclusion.call(module)) {
      logger.debug("Checking whether version ${module.version} of dependency ${module.group}:${module.name} is compatible with Prelease: '$projectPreleaseComponent'")
      String dependencyPrereleaseComponent = getPrerelease(module.version)
      if (CheckDependencyVersions.ORDERED_PRERELEASES.indexOf(dependencyPrereleaseComponent) <
        CheckDependencyVersions.ORDERED_PRERELEASES.indexOf(projectPreleaseComponent)) {
        throw new GradleException("Cannot release ${currentProject.displayName} version '${projectVersion}' with dependency ${module.group}:${module.name}:${module.version}")
      }
    }
  }

  @TaskAction
  void check() {
    def projectVersion = project.version.toString()
    def projectPreleaseComponent = getPrerelease(projectVersion)
    logger.lifecycle("Checking dependency versions for Prelease: '$projectPreleaseComponent'. Please wait, this may take a while...")

    projects.each { currentProject ->
      def exclusionChecker = dependencyExcludedChecker(currentProject)

      logger.info("Checking dependency versions for ${currentProject.displayName}")

      def configurations = currentProject.configurations.matching(configurationFilter(project))
      configurations.all(workaround)
      configurations.all(new Action<Configuration>() {
        @Override
        void execute(Configuration files) {
          if (canBeResolved(files)) {
            files.getIncoming().resolutionResult.allDependencies.each { dependency ->
              switch (dependency) {
                case ResolvedDependencyResult:
                  handleResolvedDependency(currentProject, dependency as ResolvedDependencyResult, exclusionChecker.checker)
                  break
                case UnresolvedDependencyResult:
                  throw (dependency as UnresolvedDependencyResult).getFailure()
                default:
                  throw new GradleException("Illegal state. DependencyResult was neither resolved nor unresolved.")
              }
            }
          }
        }
      })

      if (!exclusionChecker.excluded.isEmpty()) {
        logger.info("Dependencies excluded from version check: [\n${exclusionChecker.excluded.join(",\n")}\n]\n")
      }
    }
  }
}
