package com.xebialabs.gradle.artifact

import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.artifacts.component.ModuleComponentSelector
import org.gradle.api.artifacts.result.ResolvedDependencyResult
import org.gradle.api.tasks.Upload

class XLArtifactPlugin implements Plugin<Project> {

  def nexusUrl(project) {
    if (project.version.toString().contains("alpha")) {
      "${project.property("nexusBaseUrl")}/repositories/alphas"
    } else {
      "${project.property("nexusBaseUrl")}/repositories/releases"
    }
  }

  @Override
  void apply(Project project) {
    project.apply plugin: "maven"

    project.tasks.withType(Upload, { t ->
      if (t.name == 'install') {
        project.logger.info("Fixing dependencies in pom's for mavenInstall $t")
        fixDependenciesOnDeployer(project, t.repositories.mavenInstaller)
        configurePomLicense(project, t.repositories.mavenInstaller)
      } else if (t.name.startsWith('upload')) {
        project.logger.info("Configuring mavenDeployer on $t")
        t.repositories.mavenDeployer {
          if (project.hasProperty("localMavenRepository")) {
            def mavenrepo = new File(project.getRootDir(), "mavenrepo")
            mavenrepo.mkdir()
            def mavenrepoReleases = new File(mavenrepo, "releases")
            mavenrepoReleases.mkdir()
            def mavenrepoSnapshots = new File(mavenrepo, "snapshots")
            mavenrepoSnapshots.mkdir()

            repository(url: "${mavenrepoReleases.toURI().toURL()}")
            snapshotRepository(url: "${mavenrepoSnapshots.toURI().toURL()}")
          } else {
            repository(url: "${nexusUrl(project)}") {
              authentication(userName: "${project.property("nexusUserName")}", password: "${project.property("nexusPassword")}")
            }
            snapshotRepository(url: "${project.property("nexusBaseUrl")}/repositories/snapshots") {
              authentication(userName: "${project.property("nexusUserName")}", password: "${project.property("nexusPassword")}")
            }
          }
        }
        project.logger.info("Fixing dependencies in pom's for mavenDeployer $t")
        fixDependenciesOnDeployer(project, t.repositories.mavenDeployer)
        configurePomLicense(project, t.repositories.mavenDeployer)
      } else {
        project.logger.warn("Upload task could not apply pom fix to '$t' hope that's ok.")
      }
    })
  }

  // Work around: https://issues.gradle.org/browse/GRADLE-3120
  // fix the versions in the pom to match the resolved configurations
  def fixDependenciesOnDeployer(project, deployer) {
    deployer.findAll { it != null }*.pom*.whenConfigured { pom ->
      def dependencyMap = [:]
      if (project.configurations.findByName("runtime")) {
        dependencyMap['runtime'] = project.configurations.runtime.incoming.resolutionResult.allDependencies
      }
      if (project.configurations.findByName("testRuntime")) {
        dependencyMap['test'] = project.configurations.testRuntime.incoming.resolutionResult.allDependencies - dependencyMap['runtime']
      }
      pom.dependencies.each { dep ->
        def group = dep.groupId
        def name = dep.artifactId
        def scope = dep.scope

        if (['provided', 'compile'].contains(scope)) {
          scope = 'runtime'
        }

        ResolvedDependencyResult resolved = dependencyMap[scope].find { r ->
          (r.requested instanceof ModuleComponentSelector) &&
            (r.requested.group == group) &&
            (r.requested.module == name)
        }

        if (!resolved) {
          return  // continue loop if a dependency is not found in dependencyMap
        }

        dep.version = resolved?.selected?.moduleVersion?.version
      }
    }
  }

  def configurePomLicense(project, deployer) {
    deployer.findAll { it != null }*.pom*.whenConfigured { pom ->
      pom.project {
        project.logger.info("Configuring 2 EULA license for ${project.name}")
        licenses {
          license {
            name 'EULA Artifacts XebiaLabs, Version 1.0'
            url 'https://dist.xebialabs.com/public/legal/eula-artifacts-v10.txt'
            distribution 'repo'
          }
        }
      }
    }
  }

}
