package com.xebialabs.gradle.plugins

import com.xebialabs.gradle.plugins.tasks.PluginVersion
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.file.DuplicatesStrategy
import org.gradle.api.file.RegularFile
import org.gradle.api.plugins.BasePlugin
import org.gradle.api.tasks.bundling.Jar
import org.gradle.api.tasks.bundling.Zip

class XLPluginBundlePlugin implements Plugin<Project> {
  public static final String PLUGIN_VERSION_TASK_NAME = "createPluginVersionProperties"
  public static final String BUNDLING_TASK_NAME = "xlPlugin"

  @Override
  void apply(Project project) {
    addVersionPropertiesTask(project)
    setupDependencyBundling(project)
  }

  void addVersionPropertiesTask(Project project) {
    def pluginVersionTask = project.tasks.create(PLUGIN_VERSION_TASK_NAME, PluginVersion).configure {
      group = BasePlugin.BUILD_GROUP
      description = "Create a plugin-version.properties file with plugin metadata."
      conventionMapping.plugin = { -> project.name }
      conventionMapping.version = { -> project.version.toString() }
      RegularFile regularFile = project.layout.buildDirectory.file("generated/plugin-version/plugin-version.properties").get()
      pluginVersionFile = regularFile.getAsFile()
    }

    project.tasks.withType(Jar).configureEach { Jar jarTask ->
      jarTask.with {
        inputs.property('project.version', project.version.toString())
        into("") {
         from pluginVersionTask
        }
        // we want to use DuplicatesStrategy.WARN as plugin-version.properties might already be defined in a project
        duplicatesStrategy = DuplicatesStrategy.WARN
      }
    }

  }

  static def setupDependencyBundling(Project project) {
    def distBundleConfig = project.configurations.maybeCreate("distBundle")

    def xlPluginTask = project.tasks.create(BUNDLING_TASK_NAME, Zip).configure {
      group = 'Archive'
      description = "Create an archive that contains plugin jar and all dependencies defined by 'distBundle' configuration"
      duplicatesStrategy = DuplicatesStrategy.EXCLUDE

      into(".") {
        from distBundleConfig
        from project.tasks.getByName("jar")
      }
    }

    project.artifacts {
      archives xlPluginTask
    }
  }
}
