package com.xebialabs.xlplatform.extensions.exportcis.archive

import java.io._
import java.util.zip.{ZipEntry, ZipOutputStream}

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.repository.{RepositoryService, WorkDir}
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.xlplatform.extensions.exportcis.archive.RepositoryExporter._
import com.xebialabs.xlplatform.utils.ResourceManagement._
import org.apache.commons.lang.CharEncoding
import org.joda.time.DateTime
import org.joda.time.format.DateTimeFormat

import scala.language.implicitConversions
import scala.util.Try
import scalax.file.defaultfs.DefaultPath

object RepositoryExporter {
  val CiXmlFileName = "configuration-items.xml"
  val RepositoryRootAlias = "/"
}

class RepositoryExporter(val repositoryService: RepositoryService, val ctx: ExecutionContext) {

  private val xmlExporter = new RepositoryXmlExporter(repositoryService, ctx)

  def export(ciRoots: Iterable[String], exportFile: DefaultPath, userName: String, appVersion: String): Try[Unit] = {

    Try {
      val result: RepositoryXmlExporterResult = xmlExporter.exportCiTrees(ciRoots, userName, appVersion)

      ctx.logOutput(s"Started packing CIs metadata and contents into a ZIP file: $exportFile")

      using(new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(exportFile.jfile)))) { zos =>
        zos.putNextEntry(new ZipEntry(CiXmlFileName))
        zos.write(result.xmlContent.getBytes(CharEncoding.UTF_8))
        zos.closeEntry()
        for ((path, overthereContent) <- result.filesByZipEntry) {

          zos.putNextEntry(new ZipEntry(path))

          overthereContent.isDirectory match {
            case false =>
              using(overthereContent.getInputStream) { fileStream =>
                OverthereUtils.write(fileStream, zos)
              }
            case true =>
              writeFolderAsZip(new File(overthereContent.getPath), zos)
          }

          zos.closeEntry()
        }

        ctx.logOutput(s"Finished packing CIs metadata and contents into a ZIP file: $exportFile")
      }
    }
  }

  private def createWorkDir(basePath: DefaultPath): WorkDir = {
    import scalax.file.ImplicitConversions.defaultPath2jfile
    val date = DateTimeFormat.forPattern("yyyyMMdd_HHmmss_sss").print(DateTime.now)
    val workDirName = s"workdir-$date.tmp"
    val workDir = basePath / workDirName createDirectory (failIfExists = false)
    new WorkDir(LocalFile.valueOf(workDir).asInstanceOf[LocalFile])
  }


  private[archive] def writeFolderAsZip(folder: File, os: OutputStream): Unit = {

    def writeIntoZip(file: File, zos: ZipOutputStream, basePath: String = ""): Unit = {
      if (!file.isDirectory) {
        zos.putNextEntry(new ZipEntry(s"$basePath${file.getName}"))
        using(new FileInputStream(file)) { fis =>
          OverthereUtils.write(fis, zos)
        }
        zos.closeEntry()
      } else {
        for (child <- file.listFiles()) {
          writeIntoZip(child, zos, s"$basePath${file.getName}/")
        }
      }
    }

    val folderAsZipStream = new ZipOutputStream(os)
    for (file <- folder.listFiles()) {
      writeIntoZip(file, folderAsZipStream)
    }
    // Avoid closing the parent output stream
    folderAsZipStream.finish()
  }

}
