package com.xebialabs.xlplatform.extensions.exportcis.archive

import java.io.{File, FileOutputStream}
import java.nio.file.Paths
import java.security.InvalidKeyException
import java.util.zip.ZipFile

import com.thoughtworks.xstream.io.xml.JDom2Reader
import com.typesafe.config.ConfigFactory
import com.xebialabs.deployit.core.xml.PasswordEncryptingCiConverter
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.repository._
import com.xebialabs.deployit.repository.internal.Root
import com.xebialabs.deployit.service.validation.Validator
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.xlplatform.artifact.ExternalArtifactDownloader.downloadResolvedFile
import com.xebialabs.xlplatform.artifact.resolution.ArtifactResolverRegistry.{isExternalArtifact, isStoredArtifact}
import com.xebialabs.xlplatform.config.ConfigurationHolder
import com.xebialabs.xlplatform.extensions.exportcis.archive.RepositoryExporter._
import com.xebialabs.xlplatform.extensions.exportcis.archive.RepositoryImporter._
import com.xebialabs.xlplatform.extensions.exportcis.archive.RepositoryXmlExporter.Attributes._
import com.xebialabs.xlplatform.extensions.exportcis.archive.RepositoryXmlExporter.Elements._
import com.xebialabs.xlplatform.extensions.exportcis.archive.RepositoryXmlExporter._
import com.xebialabs.xlplatform.sugar.PathSugar._
import com.xebialabs.xlplatform.utils.ResourceManagement._
import com.xebialabs.xltype.serialization.CiReader
import com.xebialabs.xltype.serialization.xstream.CiXstreamReader
import grizzled.slf4j.Logging
import org.jdom2.Element
import org.jdom2.input.SAXBuilder

import scala.collection.convert.ImplicitConversions._
import scala.util.Try


object RepositoryImporter {

  val workDirFactory = new WorkDirFactory(
    ConfigFactory.load(ConfigurationHolder.get().withFallback(ConfigFactory.defaultReference())).getString("xl.export-cis.import-work-dir")
  )

  implicit class RepositoryWithCleanup(val repositoryService: RepositoryService) {

    def deleteBeforeImport(path: String) = path match {
      case "" | RepositoryRootAlias => deleteContentsOf(RepositoryRootAlias)
      case p if isInternalRoot(repositoryService.read[ConfigurationItem](path, 0).getType) =>
        deleteContentsOf(p)
      case p =>
        repositoryService.delete(p)
    }

    private def isInternalRoot(t: Type) = t.instanceOf(Type.valueOf(classOf[Root]))

    private def deleteContentsOf(path: String): Unit = {
      val nonRoots = repositoryService.list(new SearchParameters().setParent(path)).filter {
        case ciData if isInternalRoot(ciData.getType) =>
          deleteContentsOf(ciData.getId)
          false
        case ciData => true
      }

      repositoryService.delete(nonRoots.map(_.getId): _*)
    }

  }

}

class RepositoryImporter(val repositoryService: RepositoryService, val ctx: ExecutionContext) extends Logging {

  def `import`(archivePath: File): Try[Unit] = Try {

    implicit val namespace = XlNamespace
    using(new ZipFile(archivePath)) { zipArchive =>
      using(zipArchive.entryStream(CiXmlFileName)) { cisXmlIs =>

        val saxBuilder: SAXBuilder = new SAXBuilder
        val rootElement: Element = saxBuilder.build(cisXmlIs).getRootElement

        val archiveFingerprint = rootElement.child(metadata, encryptionKeyFingerprint).getValue
        val currentFingerprint = PasswordEncrypter.getInstance().getKeyFingerprint

        if (currentFingerprint != archiveFingerprint) {
          throw new InvalidKeyException(s"${archivePath.getPath} was exported with different key fingerprint: $archiveFingerprint. Fingerprint of this server: $currentFingerprint")
        }

        ctx.logOutput(s"Starting repository import from [${archivePath.getPath}]")
        val exportedCiSections = rootElement.getChildren(exportedConfigurationItems, XlNamespace)

        val joined = exportedCiSections.fold(new Element(exportedRootId))((newElement, sectionElement) => {
          newElement.addContent(sectionElement.removeContent())
        })

        ctx.logOutput(s"Converting [${joined.getContentSize}] XML nodes into CIs")
        val exportedCisReader = new JDom2Reader(joined)

        val converter = new ArtifactExtractingCiConverter(zipArchive, exportedCisReader)

        val cis = converter.readCis(new CiXstreamReader(exportedCisReader))
        ctx.logOutput(s"Finished deserialization of [${cis.size()}] CIs.")
        converter.resolveReferences(new RepositoryAdapter(repositoryService, new Validator(), null))
        ctx.logOutput(s"Finished references resolution.")

        exportedCiSections
          .map(_.getAttribute(exportedRootId).getValue)
          .foreach {
            case sectionRoot =>
              repositoryService.deleteBeforeImport(sectionRoot)
              ctx.logOutput(s"Preparing [$sectionRoot] for import")
          }

        ctx.logOutput(s"Storing CIs in the repository.")
        repositoryService.create(cis: _*)
        ctx.logOutput(s"Successfully stored ${cis.size()} in the repository.")
      }
    }
  }

  private class ArtifactExtractingCiConverter(zip: ZipFile, outerReader: JDom2Reader) extends PasswordEncryptingCiConverter {

    override def readCi(reader: CiReader): ConfigurationItem = {
      val workDir = WorkDirContext.get()
      trace(s"Processing [${reader.getId}]")
      super.readCi(reader) match {
        case artifact: SourceArtifact if isStoredArtifact(artifact) =>
          val artifactInArchive = outerReader.getAttribute("file")
          trace(s"Attaching artifact file [$artifactInArchive}] to [${reader.getId}].")
          artifact.setFile(extractedFile(artifactInArchive, zip, workDir))
          artifact
        case artifact: SourceArtifact if isExternalArtifact(artifact) =>
          artifact.setFile(downloadResolvedFile(artifact, workDir))
          artifact
        case ci => ci
      }
    }

  }

  private def extractedFile(entryPath: String, archive: ZipFile, workDir: WorkDir): OverthereFile = {

    val artifactWorkPath = Paths.get(workDir.getPath) / entryPath createFile(createParents = true, failIfExists = false)

    using(archive.entryStream(entryPath)) { entryStream =>
      using(new FileOutputStream(artifactWorkPath)) { aos =>
        OverthereUtils.write(entryStream, aos)
      }
    }

    LocalFile.valueOf(artifactWorkPath)
  }

  private[archive] def getWorkDirFactory = workDirFactory
}
