package com.xebialabs.satellite.serialization

import java.io.ByteArrayOutputStream

import akka.actor.ExtendedActorSystem
import com.esotericsoftware.kryo.io.{Input, Output}
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener
import com.xebialabs.deployit.engine.tasker.satellite.KryoSerializer
import com.xebialabs.deployit.engine.tasker.{StepBlock, TaskSpecification}
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact
import com.xebialabs.xlplatform.satellite.SatelliteAware
import nl.javadude.scannit.Scannit

import scala.collection.convert.wrapAll._
import scala.util.Try

object TaskSpecificationSerialization {

  def apply(extendedActorSystem: ExtendedActorSystem): TaskSpecificationSerialization = new KryoSerialization {
    val system = extendedActorSystem
  }

  lazy val artifactTypes = Scannit.getInstance().getSubTypesOf(classOf[Artifact]).toList
  lazy val stepTypes = Scannit.getInstance().getSubTypesOf(classOf[SatelliteAware]).toList

  trait KryoSerialization extends TaskSpecificationSerialization {

    def system: ExtendedActorSystem

    override def fromBinary(bytes: Array[Byte], files: UploadedFiles, artifactImplems: Seq[Class[_ <: Artifact]], stepImplems: Seq[Class[_ <: Step]]) = {
      Try {
        val kryo = KryoSerializer.getMagicCombination(system)

        kryo.register(classOf[StepBlock], new StepBlockSerializer(kryo))

        kryo.addDefaultSerializer(classOf[ExecutionStateListener], new ExecutionStateListenerSerializer(kryo))

        artifactImplems.foreach(clazz => {
          kryo.register(clazz, new ArtifactSerializer(kryo, clazz, Right(files)))
        })

        stepImplems.foreach(clazz => {
          kryo.register(clazz, new StepSerializer(kryo, clazz))
        })

        val input: Input = new Input(bytes)

        val deserialized = kryo.readObject(input, classOf[TaskSpecification])

        input.close()

        val listeners = deserialized.getListeners.filterNot(_ == null).toList

        deserialized.getListeners.removeAll(deserialized.getListeners)
        deserialized.getListeners.addAll(listeners)


        deserialized
      }
    }

    override def toBinary(obj: TaskSpecification): Try[(Array[Byte], FilesToUpload, Seq[Class[_ <: Artifact]], Seq[Class[_ <: Step]])] = {
      val files = FilesCache.empty

      val kryo = KryoSerializer.getMagicCombination(system)

      kryo.register(classOf[StepBlock], new StepBlockSerializer(kryo))

      kryo.addDefaultSerializer(classOf[ExecutionStateListener], new ExecutionStateListenerSerializer(kryo))

      artifactTypes.foreach(clazz => {
        kryo.register(clazz, new ArtifactSerializer(kryo, clazz, Left(files)))
      })

      stepTypes.foreach(clazz => {
        kryo.register(clazz, new StepSerializer(kryo, clazz))
      })


      Try {
        val outputStream = new ByteArrayOutputStream()
        val output = new Output(outputStream)
        kryo.writeObject(output, obj)

        output.flush()
        output.close()

        (outputStream.toByteArray, files.toSet, artifactTypes, stepTypes)
      }
    }
  }

}

trait TaskSpecificationSerialization {

  def toBinary(obj: TaskSpecification): Try[(Array[Byte], FilesToUpload, Seq[Class[_ <: Artifact]], Seq[Class[_ <: Step]])]

  def fromBinary(bytes: Array[Byte],
                 files: UploadedFiles,
                 artifactImplems: Seq[Class[_ <: Artifact]],
                 stepImplems: Seq[Class[_ <: Step]]): Try[TaskSpecification]

}