package com.xebialabs.satellite.streaming

import java.util.zip.{Deflater, Inflater}

import akka.util.{ByteString, ByteStringBuilder}

import scala.collection.mutable

object StreamOps {

  private def zipCompressor = new ZipCompressor

  val compress = { bytes: ByteString =>
    val b = new ByteStringBuilder

    b ++= zipCompressor.compress(bytes.toArray)

    b.result()
  }

  val decompress = { bytes: ByteString =>
    val b = new ByteStringBuilder

    b ++= zipCompressor.decompress(bytes.toArray)

    b.result()
  }

  val toByteString = { bytes: Seq[Byte] =>
    val b = new ByteStringBuilder
    b.sizeHint(bytes.length)
    b ++= bytes
    b.result()
  }

  class ZipCompressor {

    lazy val deflater = new Deflater(Deflater.BEST_SPEED)
    lazy val inflater = new Inflater()

    def compress(inputBuff: Array[Byte]): Array[Byte] = {
      val inputSize = inputBuff.length
      val outputBuff = new mutable.ArrayBuilder.ofByte
      outputBuff += (inputSize & 0xff).toByte
      outputBuff += (inputSize >> 8 & 0xff).toByte
      outputBuff += (inputSize >> 16 & 0xff).toByte
      outputBuff += (inputSize >> 24 & 0xff).toByte

      deflater.setInput(inputBuff)
      deflater.finish()
      val buff = new Array[Byte](4096)

      while (!deflater.finished) {
        val n = deflater.deflate(buff)
        outputBuff ++= buff.take(n)
      }
      deflater.reset()
      outputBuff.result()
    }

    def decompress(inputBuff: Array[Byte]): Array[Byte] = {
      val size: Int = (inputBuff(0).asInstanceOf[Int] & 0xff) |
        (inputBuff(1).asInstanceOf[Int] & 0xff) << 8 |
        (inputBuff(2).asInstanceOf[Int] & 0xff) << 16 |
        (inputBuff(3).asInstanceOf[Int] & 0xff) << 24
      val outputBuff = new Array[Byte](size)
      inflater.setInput(inputBuff, 4, inputBuff.length - 4)
      inflater.inflate(outputBuff)
      inflater.reset()
      outputBuff
    }
  }


}
