package com.xebialabs.xlplatform.settings

import java.io.File
import java.util.concurrent.TimeUnit

import akka.actor.{ExtendedActorSystem, Extension, ExtensionId, ExtensionIdProvider}
import akka.util.Timeout
import com.typesafe.config.Config

import scala.concurrent.duration._

object CommonSettings extends ExtensionId[CommonSettings] with ExtensionIdProvider {

  override def createExtension(system: ExtendedActorSystem): CommonSettings = new CommonSettings(system.settings.config)

  override def lookup(): ExtensionId[_ <: Extension] = CommonSettings
}

class CommonSettings(config: Config) extends Extension {
  lazy val tasker = new TaskerSettings(config)
  lazy val satellite = new SatelliteSettings(config)
  lazy val scheduler = new SchedulerSettings(config)
  lazy val security = new SecuritySettings(config.getConfig("satellite.ssl"))
}

class TaskerSettings(val config: Config) extends SettingsSupport {
  lazy val recoveryDir = new File(value[String]("task.recovery-dir"))
  lazy val maxActiveTasks: Int = value[Int]("task.max-active-tasks")
  lazy val shutdownTimeout: FiniteDuration = value[FiniteDuration]("task.shutdown-timeout")
  lazy val tickDuration: FiniteDuration = value[FiniteDuration]("akka.scheduler.tick-duration")
  lazy val stepRetryDelay: FiniteDuration = value[FiniteDuration]("task.step.retry-delay")
  lazy val askTimeout: Timeout = valueWithDefault("tasker.askTimeout", FiniteDuration(10, TimeUnit.SECONDS))
}

class SatelliteSettings(val config: Config) extends SettingsSupport {
  lazy val uploadIdleTimeout: Timeout = valueWithDefault("satellite.timeout.upload.idle", FiniteDuration(10, TimeUnit.SECONDS))
  lazy val pingTimeout: Timeout = valueWithDefault("satellite.timeout.ping", FiniteDuration(5, TimeUnit.SECONDS))
  lazy val remoteAskTimeout: Timeout = valueWithDefault("satellite.timeout.remoteAsk", FiniteDuration(60, TimeUnit.SECONDS))
  lazy val hostname: String = value[String]("satellite.hostname")
  lazy val enabled: Boolean = valueWithDefault("satellite.enabled", false)
  lazy val streamingConnectionTimeout: Timeout = valueWithDefault("satellite.timeout.streaming", FiniteDuration(10, TimeUnit.SECONDS))
  lazy val maxConcurrentUploads: Int = valueWithDefault("satellite.streaming.max-uploads", 10)
}

class SchedulerSettings(val config: Config) extends SettingsSupport {
  lazy val tickDuration: FiniteDuration = value[FiniteDuration]("akka.scheduler.tick-duration")
}