package com.xebialabs.deployit.engine.tasker

import java.util.UUID

import akka.actor._
import akka.event.LoggingReceive
import com.xebialabs.deployit.engine.api.execution.{BlockExecutionState, StepExecutionState}
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor._
import com.xebialabs.deployit.engine.tasker.messages._
import com.xebialabs.deployit.engine.tasker.satellite.{Paths, ActorLocator}

class BlockOnSatellite private(taskId: TaskId, block: ExecutableBlock, actorLocator: ActorLocator, notificationActor: ActorRef)
  extends Actor with ModifyStepsSupport with Stash {

  private val remoteActor = actorLocator.locate(Paths.tasks)(context.system)

  override def receive: Receive = disconnected

  def disconnected: Receive = {
    case _ =>
      tryConnect()
      stash()
  }

  private def tryConnect() = {
    val uuid = UUID.randomUUID()
    remoteActor ! Identify(uuid)
    context become identifyRemoteActor(uuid, sender())
  }

  private def identifyRemoteActor(uuid: UUID, originalSender: ActorRef): Receive = {
    case ActorIdentity(`uuid`, Some(actorRef)) =>
      debug(s"Remote actor $remoteActor found")
      context.watch(actorRef)
      unstashAll()
      context become (sendModifyStepsAndUpdateBlock(actorRef) orElse sendToSatellite(actorRef) orElse handleBlockDoneOrChanged orElse handleDeathOfRemote())
    case ActorIdentity(`uuid`, None) =>
      disconnect(s"Could not connect to $remoteActor. Probably, connection to a satellite is broken", Option(originalSender))
    case _ =>
      stash()
  }

  private def handleDeathOfRemote(originalSender: Option[ActorRef] = None): Receive = {
    case Terminated(actorRef) =>
      context.unwatch(actorRef)
      disconnect(s"Remote actor $remoteActor is terminated. Probably, connection to a satellite is broken", originalSender)
  }

  private def disconnect(msg: => String, originalSender: Option[ActorRef] = None) {
    warn(msg)
    context become disconnected
    originalSender.foreach(_.tell(ActorNotFound(remoteActor), self))
    markBlockAsFailed(block)
    notificationActor ! BlockDone(taskId, block)
  }

  private def sendToSatellite(remoteActor: ActorRef): Receive = LoggingReceive {
    case msg@Start(`taskId`) =>
      debug(s"sending $msg to remote")
      remoteActor ! StartBlock(taskId, block.id)
    case msg@Stop(`taskId`) =>
      debug(s"sending $msg to remote")
      remoteActor ! StopBlock(taskId, block.id)
    case msg@Abort(`taskId`) =>
      debug(s"sending $msg to remote")
      remoteActor ! AbortBlock(taskId, block.id)
  }

  private def handleBlockDoneOrChanged: Receive = {
    case BlockDone(`taskId`, updatedBlock: ExecutableBlock) =>
      updateState(block, updatedBlock)
      notificationActor ! BlockDone(taskId, block)
    case BlockStateChanged(`taskId`, updatedBlock: ExecutableBlock, oldState, newState) =>
      updateState(block, updatedBlock)
      notificationActor ! BlockStateChanged(taskId, block, oldState, newState)
  }

  private def updateState(localBlock: ExecutableBlock, remoteBlock: ExecutableBlock) {
    (localBlock, remoteBlock) match {
      case (local: StepBlock, remote: StepBlock) =>
        local.newState(remote.state)
        local.steps = remote.steps

      case (local: CompositeBlock, remote: CompositeBlock) =>
        local.blocks.zip(remote.blocks).foreach {
          case (localSubBlock, remoteSubBlock) =>
            local.newState(remote.state)
            updateState(localSubBlock, remoteSubBlock)
        }
      case _ =>
    }
  }

  private def markBlockAsFailed(localBlock: ExecutableBlock): Unit = localBlock match {
    case cb: CompositeBlock =>
      if (!cb.state.isFinished) {
        cb.newState(BlockExecutionState.FAILED)
      }
      cb.blocks.foreach(markBlockAsFailed)
    case st: StepBlock =>
      st.newState(BlockExecutionState.FAILED)
      st.steps.collect { case ts: TaskStep => ts}.filterNot(_.getState.isFinished).foreach(_.setState(StepExecutionState.FAILED))
  }

  private def sendModifyStepsAndUpdateBlock(actorRef: ActorRef) : Receive = {
    case msg: ModifySteps if msg.taskId == taskId =>
      debug(s"sending $msg to the remote actor")
      context.become(handleDeathOfRemote(Option(sender())) orElse waitForStepModified(taskId, block, context.sender(), msg), discardOld = false)
      actorRef ! BlockEnvelope(taskId = msg.taskId, blockPath = block.id, message = msg)
  }

  private def waitForStepModified(taskId: TaskId, block: ExecutableBlock, originalSender: ActorRef, originalMessage: ModifySteps) : Receive = {
    case msg: ErrorMessage =>
      originalSender ! msg
      context.unbecome()
      unstashAll()
    case msg: SuccessMessage =>
      modifySteps(taskId, path => path.relative(block.id).flatMap(block.getBlock), () => originalSender).apply(originalMessage)
      context.unbecome()
      unstashAll()
    case _ => stash()
  }
}

object BlockOnSatellite {

  def props(taskId: TaskId, block: ExecutableBlock, actorLocator: ActorLocator, notificationActor: ActorRef): Props =
    Props(classOf[BlockOnSatellite], taskId, block, actorLocator, notificationActor)
}